/*****************************************************************************
** File:        ay8910.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the AY8910 sound chip
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
** History:     1.0 - 10/17 2003 Initial version
** History:     1.1 - 11/21 2003 Updated phase handling (bug in Chuckie Egg)
**              1.2 - 12/6  2003 Fixed problem with playing samples
**
******************************************************************************
*/
#include "AY8910.h"
#include <stdlib.h>
#include <string.h>

#define BASE_PHASE_STEP 0x28959becUL  /* = (1 << 28) * 3579545 / 32 / 44100 */
#define BUFFER_SIZE     1024

static Int16 voltTable[16] = {
    0x0000, 0x004f, 0x00b4, 0x0133, 0x01d4, 0x029f, 0x03a1, 0x04e7, 
    0x0683, 0x088c, 0x0b1f, 0x0e62, 0x1281, 0x17b8, 0x1e50, 0x26a9
};

static Int16* ay8910Sync(void* ref, UInt32 count);

struct AY8910 {
    Mixer* mixer;

    UInt8  address;
    UInt8  regs[16];

    Int16  buffer[BUFFER_SIZE];

    UInt32 tonePhase[3];
    UInt32 toneStep[3];

    UInt32 noisePhase;
    UInt32 noiseStep;
    UInt32 noiseRand;
    Int16  noiseVolume;

    UInt8  envShape;
    UInt32 envStep;
    UInt32 envPhase;

    UInt8  enable;
    UInt8  ampVolume[3];
    Int32  daVolume;
};

int ay8910GetSize() {
    return sizeof(AY8910);
}

AY8910* ay8910Create(Mixer* mixer, UInt32 cpuClock)
{
    AY8910* ay8910 = (AY8910*)calloc(1, sizeof(AY8910));
    int i;

    ay8910->mixer = mixer;
    ay8910->noiseRand = 1;
    ay8910->noiseVolume = 1;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_PSG, ay8910Sync, ay8910);

    for (i = 0; i < 16; i++) {
        ay8910WriteAddress(ay8910, i);
        ay8910WriteData(ay8910, 0, cpuClock);
    }

    return ay8910;
}

void ay8910Destroy(AY8910* ay8910)
{
    mixerUnregisterChannel(ay8910->mixer, MIXER_CHANNEL_PSG);
    free(ay8910);
}

UInt8 ay8910GetLatch(AY8910* ay8910) {
    return ay8910->address;
}

UInt8 ay8910GetRegister(AY8910* ay8910, UInt8 address) {
    return ay8910->regs[address];
}

void ay8910WriteAddress(AY8910* ay8910, UInt8 address)
{
    ay8910->address = address & 0xf;
}

UInt8 ay8910ReadData(AY8910* ay8910)
{
    return ay8910->regs[ay8910->address];
}

void ay8910WriteData(AY8910* ay8910, UInt8 data, UInt32 cpuClock)
{
    UInt8  address = ay8910->address;
    UInt32 period;

    mixerSync(ay8910->mixer, cpuClock);

    ay8910->regs[address] = data;

    switch (address) {
    case 0:
    case 1:
    case 2:
    case 3:
    case 4:
    case 5:
        period = ay8910->regs[address & 6] | ((Int32)(ay8910->regs[address | 1] & 0x0f) << 8);
        ay8910->toneStep[address >> 1] = period > 1 ? BASE_PHASE_STEP / period : 0;
        if (period <= 1) ay8910->tonePhase[address >> 1] = 0;
        break;
        
    case 6:
        period = data & 0x1f ? data & 0x1f : 1;
        ay8910->noiseStep = BASE_PHASE_STEP / period;
        break;
        
    case 7:
        ay8910->enable = data & 0x3f;
        break;
        
    case 8:
    case 9:
    case 10:
        ay8910->ampVolume[address - 8] = data & 0x1f;
        break;

    case 11:
    case 12:
        period = 16 * (ay8910->regs[11] | ((UInt32)ay8910->regs[12] << 8));
        ay8910->envStep = BASE_PHASE_STEP / (period ? period : 8);
        break;
        
    case 13:
        if (data < 4) data = 0x09;
        if (data < 8) data = 0x0f;
        ay8910->envShape = data;
        ay8910->envPhase = 0;
        break;
    }
}

static Int16* ay8910Sync(void* ref, UInt32 count)
{
    AY8910* ay8910 = (AY8910*)ref;
    Int32   channel;
    UInt32  index;

    for (index = 0; index < count; index++) {
        Int16 envVolume;
        Int16 sampleVolume = 0;

        /* Update noise generator */
        ay8910->noisePhase += ay8910->noiseStep;
        while (ay8910->noisePhase >> 28) {
            ay8910->noisePhase  -= 0x10000000;
            ay8910->noiseVolume ^= ((ay8910->noiseRand + 1) >> 1) & 1;
            ay8910->noiseRand    = (ay8910->noiseRand ^ (0x28000 * (ay8910->noiseRand & 1))) >> 1;
        }

        /* Update envelope phase */
        ay8910->envPhase += ay8910->envStep;
        if ((ay8910->envShape & 1) && (ay8910->envPhase >> 28)) {
            ay8910->envPhase = 0x10000000;
        }
 
        /* Calculate envelope volume */
        envVolume = (Int16)((ay8910->envPhase >> 24) & 0x0f);
        if (((ay8910->envPhase >> 27) & (ay8910->envShape + 1) ^ (~ay8910->envShape >> 1)) & 2) {
            envVolume ^= 0x0f;
        }

        /* Calculate and add channel samples to buffer */
        for (channel = 0; channel < 3; channel++) {
            UInt8 enable = ay8910->enable >> channel;
            Int16 tone;

            /* Update phase of tone */
            ay8910->tonePhase[channel] += (~enable & 1) * ay8910->toneStep[channel];
 
            /* Calculate if tone is on or off */
            tone = (Int16)((enable | (~ay8910->tonePhase[channel] >> 27)) & 
                   ((enable >> 3) | ay8910->noiseVolume) & 1);

            /* Amplify sample using either envelope volume or channel volume */
            if (ay8910->ampVolume[channel] & 0x10) {
                sampleVolume += tone * voltTable[envVolume];
            }
            else {
                sampleVolume += tone * voltTable[ay8910->ampVolume[channel]];
            }
        }

        /* Emulate DA converter */
        ay8910->daVolume -= 2 * (ay8910->daVolume - sampleVolume) / 3;
        ay8910->buffer[index] = (Int16)ay8910->daVolume;
    }

    return ay8910->buffer;
}
