/*****************************************************************************
** File:        scc.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the Konami SCC sound chip
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
** History:     1.0 - 10/17 2003 Initial version
**              1.1 - 11/20 2003 Improved audio when channels are turned on
**                               and off
**
******************************************************************************
*/
#include "SCC.h"
#include <stdlib.h>
#include <string.h>

#define BASE_PHASE_STEP 0x28959becUL  /* = (1 << 28) * 3579545 / 32 / 44100 */
#define BUFFER_SIZE     1024

static Int16* sccSync(void* ref, UInt32 count);

struct SCC
{
    Mixer* mixer;

    UInt32 writeIndex;
    UInt32 indexCounter;
    UInt8  regs[256];
    UInt32 phase[5];
    Int32  prevSample[5];
    Int32  nextSample[5];
    Int16  buffer[BUFFER_SIZE];
};

Int32 sccGetSize() {
    return sizeof(SCC);
}

SCC* sccCreate(Mixer* mixer, UInt32 cpuClock)
{
    SCC* scc = (SCC*)calloc(1, sizeof(SCC));

    scc->mixer = mixer;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_SCC, sccSync, scc);

    return scc;
}

void sccDestroy(SCC* scc)
{
    mixerUnregisterChannel(scc->mixer, MIXER_CHANNEL_SCC);
    free(scc);
}

UInt8 sccRead(SCC* scc, UInt8 address)
{
    if (address >= 0x80) {
        return 0xff;
    }

    return scc->regs[address];
}

UInt8 sccPlusRead(SCC* scc, UInt8 address)
{
    if (address >= 0xa0) {
        return 0xff;
    }

    return scc->regs[address];
}

void sccWrite(SCC* scc, UInt8 address, UInt8 value, UInt32 cpuClock)
{
    if (address >= 0xe0) {
        return;
    }
    
    mixerSync(scc->mixer, cpuClock);

    if (address >= 0xa0) {
        address &= 0xef;
    }

    if (address >= 0x60) { 
        scc->regs[address + 0x20] = value;
    }

    if(address < 0x80) {
        scc->regs[address] = value;
    }
}

void sccPlusWrite(SCC* scc, UInt8 address, UInt8 value, UInt32 cpuClock)
{
    mixerSync(scc->mixer, cpuClock);

    if (address >= 0xa0) {
        address &= 0xef;
    }

    scc->regs[address] = value;
}

static Int16* sccSync(void* ref, UInt32 count)
{
    SCC* scc = (SCC*)ref;
    Int16* buffer  = scc->buffer;
    Int32  channel;

    memset(buffer, 0, sizeof(Int16) * count);

    /* Add sound channel data to buffer */
    for (channel = 0; channel < 5; channel++) {
        /* Precalculate values for sample generating loop */
        Int8*  waveData    = scc->regs + 32 * channel;
        Int32  volume      = 51 * (scc->regs[0xaa + channel] & 0x0f) / 15;
        Int32  period      = ((Int32)scc->regs[0xa1 + 2 * channel] << 8) + scc->regs[0xa0 + 2 * channel];
        Int32  phaseStep   = period ? ((scc->regs[0xaf] >> channel) & 1) * BASE_PHASE_STEP / (1 + period) : 0;
        Int32  phase       = scc->phase[channel];
        Int32  sampleIndex = phase >> 23;
        Int32  prevSample  = scc->prevSample[channel];
        Int32  nextSample  = scc->nextSample[channel];
        UInt32 index;

        /* Add to output buffer using linear interpolation */
        for (index = 0; index < count; index++) {
            phase = (phase + phaseStep) & 0xfffffff;

            if ((phase >> 23) != sampleIndex) {
                prevSample = nextSample;
                sampleIndex = phase >> 23;
                nextSample = waveData[sampleIndex] * volume;
            }

            buffer[index] += (Int16)(prevSample + (phase >> 8 & 0x7fff) * (nextSample - prevSample) / 0x8000);
        }

        /* Save previous sample and phase */
        scc->prevSample[channel] = prevSample;
        scc->nextSample[channel] = nextSample;
        scc->phase[channel]      = phase;
    }

    return scc->buffer;
}
