/*****************************************************************************
** File:        arch_Win32.c
**
** Author:      Daniel Vik
**
** Description: The central file that ties everything together. Contains
**              the program entry point and the main window.
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
******************************************************************************
*/
#define DIRECTINPUT_VERSION     0x0500

#define RED_MSX_VERSION "1.2"

#include "MSX.h"
#include <iostream.h>
#include <windows.h>
#include <CommCtrl.h>
#include "resource.h"
#include <Mmsystem.h>
#include "arch_Win32Timer.h"
#include "arch_Win32Sound.h"
#include "arch_Win32Properties.h"
#include "arch_Win32joystick.h"
#include "arch_Win32directx.h"
#include "arch_Win32file.h"
#include "audioMixer.h"
#include "videoRender.h"
#include "build_number.h"
#include "romMSX.h"
#include "romMSXBR.h"
#include "romMSX2.h"
#include "romMSX2EXT.h"
#include "romMSX2P.h"
#include "romMSX2PEXT.h"
#include "romMSXJ.h"
#include "romMSX2J.h"
#include "romMSX2EXTJ.h"

#include <ddraw.h>
#include <math.h>

#ifndef PIXEL_TYPE_DEFINED
#define PIXEL_TYPE_DEFINED
typedef unsigned int pixel;
#endif


#ifndef BYTE_TYPE_DEFINED
#define BYTE_TYPE_DEFINED
typedef unsigned char byte;
#endif

extern int  SyncPeriod;
extern Mixer* TheMixer;

int ttshots;

void StartEmulator(); 
void StopEmulator();

void ScreenShot(HWND hwnd);
PBITMAPINFO CreateBitmapInfoStructure(HBITMAP hBmp);
HRESULT SaveBitmap(char strFileName[128], PBITMAPINFO pbi, HBITMAP hBMP, HDC hDC);

static DWORD WINAPI ThreadProc(void* param);
static Properties* pProperties;


typedef enum { EMU_RUNNING, EMU_PAUSED, EMU_STOPPED, EMU_SUSPENDED } EmuState;

typedef struct
{
    HWND hwnd;
    HMENU hMenu;
    int showMenu;
    HBITMAP hBmp;
    HBITMAP hLogo;
    BITMAPINFO bmInfo;
    HANDLE hThread;
    HANDLE hEvent;
    void* bmBits;
    void* bmBitsGDI;
    byte keyMap[16];
    EmuState emuState;
    void* timerId;
    int updatePending;
    HWND hwndStatus;
    char pStoredState[MAX_PATH];
    char pCurDir[MAX_PATH];
    Video* pVideo;
    int logSound;
    Mixer* mixer;
    int enteringFullscreen;
	int screenMode;
    int X;
    int Y;
    int DX;
    int DY;
    int DDY;
} Port;

#define WM_UPDATE (WM_USER + 0)
#define WM_UPDATESTATUSBAR (WM_USER + 1)

/** Various variables ****************************************/
#define WIDTH  320
#define HEIGHT 240
//#define WIDTH  272
//#define HEIGHT 228

static DWORD BPal[256], XPal[300], XPal0;
static char *XBuf;
static Port Prt;
static byte JoyState;
static int CurDisk[2];


#include "arch_Win32MouseEmu.h"

volatile int suspendFlag = 0;
void suspendEmulator() {
    if (Prt.emuState == EMU_RUNNING) {
        Prt.emuState = EMU_SUSPENDED;
        do {
            Sleep(10);
        } while (!suspendFlag);
        soundSuspend();
    }
}

void resumeEmulator() {
    if (Prt.emuState == EMU_SUSPENDED) {
        Prt.emuState = EMU_RUNNING;
        soundResume();
    }
}

void restartSound(int stereo) {
    suspendEmulator();
    soundDestroy();
    soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), stereo ? 2 : 1);
    resumeEmulator();
}


/** Keyboard bindings ****************************************/
typedef struct { 
    DWORD Code; 
    byte Pos; 
    byte Mask; 
} Keybinding;

int virtualKeyTable[256];
int scanCodeTable[256];

/* Keys that cannot be mapped by scan code using MapVirtualKey */
Keybinding KeyCodeMap[] = {
  {VK_SHIFT,     6,  0x01 }, /* SHIFT      */
  {VK_SHIFT,     6,  0x01 }, /* SHIFT      */
  {VK_CONTROL,   6,  0x02 }, /* CONTROL    */
  {VK_CONTROL,   6,  0x02 }, /* CONTROL    */
  {VK_MENU,      6,  0x04 }, /* GRAPH      */
  {VK_MENU,      6,  0x10 }, /* CODE       */
  {0,0,0x00}               /** The End. **/
};

/* Keys that can be mapped by scan code using MapVirtualKey */
Keybinding ScanCodeMap[] = {
  { 0x0B, 0,  0x01 }, /* 0          */
  { 0x02, 0,  0x02 }, /* 1          */
  { 0x03, 0,  0x04 }, /* 2          */
  { 0x04, 0,  0x08 }, /* 3          */
  { 0x05, 0,  0x10 }, /* 4          */
  { 0x06, 0,  0x20 }, /* 5          */
  { 0x07, 0,  0x40 }, /* 6          */
  { 0x08, 0,  0x80 }, /* 7          */
  { 0x09, 1,  0x01 }, /* 8          */
  { 0x0A, 1,  0x02 }, /* 9          */
  { 0x0C, 1,  0x04 }, /* -          */
  { 0x0D, 1,  0x08 }, /* =          */
  { 0x2B, 1,  0x10 }, /* \          */
  { 0x1A, 1,  0x20 }, /* [          */
  { 0x1B, 1,  0x40 }, /* ]          */
  { 0x27, 1,  0x80 }, /* ;          */
  { 0x28, 2,  0x01 }, /* '          */
  { 0x29, 2,  0x02 }, /* `          */
  { 0x33, 2,  0x04 }, /* ,          */
  { 0x34, 2,  0x08 }, /* .          */
  { 0x35, 2,  0x10 }, /* /          */
  { 0x1E, 2,  0x40 }, /* A          */
  { 0x30, 2,  0x80 }, /* B          */
  { 0x2E, 3,  0x01 }, /* C          */
  { 0x20, 3,  0x02 }, /* D          */
  { 0x12, 3,  0x04 }, /* E          */
  { 0x21, 3,  0x08 }, /* F          */
  { 0x22, 3,  0x10 }, /* G          */
  { 0x23, 3,  0x20 }, /* H          */
  { 0x17, 3,  0x40 }, /* I          */
  { 0x24, 3,  0x80 }, /* J          */
  { 0x25, 4,  0x01 }, /* K          */
  { 0x26, 4,  0x02 }, /* L          */
  { 0x32, 4,  0x04 }, /* M          */
  { 0x31, 4,  0x08 }, /* N          */
  { 0x18, 4,  0x10 }, /* O          */
  { 0x19, 4,  0x20 }, /* P          */
  { 0x10, 4,  0x40 }, /* Q          */
  { 0x13, 4,  0x80 }, /* R          */
  { 0x1F, 5,  0x01 }, /* S          */
  { 0x14, 5,  0x02 }, /* T          */
  { 0x16, 5,  0x04 }, /* U          */
  { 0x2f, 5,  0x08 }, /* V          */
  { 0x11, 5,  0x10 }, /* W          */
  { 0x2D, 5,  0x20 }, /* X          */
  { 0x15, 5,  0x40 }, /* Y          */
  { 0x2C, 5,  0x80 }, /* Z          */
  { 0x3A, 6,  0x08 }, /* CAPSLOCK   */
  { 0x51, 6,  0x10 }, /* CODE       */ /* ? */
  { 0x3B, 6,  0x20 }, /* F1         */
  { 0x3C, 6,  0x40 }, /* F2         */
  { 0x3D, 6,  0x80 }, /* F3         */
  { 0x3E, 7,  0x01 }, /* F4         */
  { 0x3F, 7,  0x02 }, /* F5         */
  { 0x01, 7,  0x04 }, /* ESCAPE     */
  { 0x0F, 7,  0x08 }, /* TAB        */
  { 0x49, 7,  0x10 }, /* STOP/BREAK */
  { 0x0E, 7,  0x20 }, /* BACKSPACE  */
  { 0x4F, 7,  0x40 }, /* SELECT     */
  { 0x1C, 7,  0x80 }, /* RETURN     */
  { 0x39, 8,  0x01 }, /* SPACE      */
  { 0x47, 8,  0x02 }, /* HOME/CLS   */
  { 0x52, 8,  0x04 }, /* INSERT     */
  { 0x53, 8,  0x08 }, /* DELETE     */
  { 0x4B, 8,  0x10 }, /* LEFT ARROW */
  { 0x48, 8,  0x20 }, /* UP ARROW   */
  { 0x4D, 8,  0x80 }, /* RIGHTARROW */
  { 0x50, 8,  0x40 }, /* DOWN ARROW */
  { 0x37, 9,  0x01 }, /* NUMPAD *   */
  { 0x4E, 9,  0x02 }, /* NUMPAD +   */
  { 0x35, 9,  0x04 }, /* NUMPAD /   */
  { 0x52, 9,  0x08 }, /* NUMPAD 0   */
  { 0x4F, 9,  0x10 }, /* NUMPAD 1   */
  { 0x50, 9,  0x20 }, /* NUMPAD 2   */
  { 0x51, 9,  0x40 }, /* NUMPAD 3   */
  { 0x4B, 9,  0x80 }, /* NUMPAD 4   */
  { 0x4C, 10, 0x01 }, /* NUMPAD 5   */
  { 0x4D, 10, 0x02 }, /* NUMPAD 6   */
  { 0x47, 10, 0x04 }, /* NUMPAD 7   */
  { 0x48, 10, 0x08 }, /* NUMPAD 8   */
  { 0x49, 10, 0x10 }, /* NUMPAD 9   */
  { 0x4A, 10, 0x20 }, /* NUMPAD -   */
  { 0x53, 10, 0x40 }, /* NUMPAD ??? */
  { 0x45, 10, 0x80 }, /* NUMPAD .   */
  { 0x00, 0,  0x00 }  /** The End. **/
};

void modifyScanCodeMap(DWORD oldCode, DWORD newCode) {
    int i;

    for (i = 0; ScanCodeMap[i].Code != 0; i++) {
        if (ScanCodeMap[i].Code == oldCode) {
            ScanCodeMap[i].Code = newCode;
        }
    }
}

void InitKeyboard() 
{    
    int i;
    BOOL success;
    char klId[KL_NAMELENGTH];

    memset(klId, 0, sizeof(klId));

    /* Modify scan code map if nessecary */
    success = GetKeyboardLayoutName(klId) ;
    if (success) {
        if (0 == strcmp(klId + 4, "0411")) {
            modifyScanCodeMap(0x2B, 0x7D);
        }
    }
    
    /* Init keyboard map tables */
    for (i = 0; i < 256; i++) {
        int j;
        
        for (j = 0; KeyCodeMap[j].Code != 0; j++) {
            if (i == KeyCodeMap[j].Code) {
                break;
            }
        }
        virtualKeyTable[i] = j;
        
        for (j = 0; ScanCodeMap[j].Code != 0; j++) {
            if (MapVirtualKey(i, 0) == ScanCodeMap[j].Code) {
                break;
            }
        }
        scanCodeTable[i] = j;
    }
}

void UpdateKeyboardState()
{
    int i;

    /* Clear keyboard state buffer */
    memset(Prt.keyMap, 0xff, sizeof(Prt.keyMap));

    for(i = 0; i < 256; i++) {
        if (GetAsyncKeyState(i) >  1UL) {
            int index = virtualKeyTable[i];
            Prt.keyMap[KeyCodeMap[index].Pos] &= ~KeyCodeMap[index].Mask;
            index = scanCodeTable[i];
            Prt.keyMap[ScanCodeMap[index].Pos] &= ~ScanCodeMap[index].Mask;
        }
    }
}

int getSyncPeriod() {
    return pProperties->emulation.syncMethod == P_EMU_SYNC1MS ? 1 : 10;
}

static int getZoom() {
    switch (pProperties->video.size) {
    case P_VIDEO_SIZEX1:
        return 1;
    case P_VIDEO_SIZEX2:
        return 2;
    case P_VIDEO_SIZEFULLSCREEN:
        switch (pProperties->video.fullRes) {
        case P_VIDEO_FRES320X240_16:
        case P_VIDEO_FRES320X240_32:
            return 1;
        case P_VIDEO_FRES640X480_16:
        case P_VIDEO_FRES640X480_32:
            return 2;
        }
    }
    return 1;
}

static void setStatusBar(int usage, int dropped) {
    RECT cr;
    int parts[4];

    GetClientRect(Prt.hwnd, &cr);
    parts[0] = cr.right - 200;
    parts[1] = cr.right - 72;
    parts[2] = cr.right - 1;

    SendMessage(Prt.hwndStatus, SB_SETPARTS, (WPARAM)3, (LPARAM) parts); 
    switch (Prt.emuState) {
    case EMU_STOPPED:
        SendMessage(Prt.hwndStatus, SB_SETTEXT, 0, (LPARAM)"  Stopped");
        break;
    case EMU_PAUSED:
        SendMessage(Prt.hwndStatus, SB_SETTEXT, 0, (LPARAM)"  Paused");
        break;
    case EMU_RUNNING:
        SendMessage(Prt.hwndStatus, SB_SETTEXT, 0, (LPARAM)"  Running");
        break;
    }

    if (dropped >= 0) {
        char pBuf[32];
        sprintf(pBuf, "  Dropped frames: %d", dropped);
        SendMessage(Prt.hwndStatus, SB_SETTEXT, 1, (LPARAM)pBuf);
    }

    if (usage > -1) {
        char pBuf[32];
        sprintf(pBuf, "  %d%%", usage);
        SendMessage(Prt.hwndStatus, SB_SETTEXT, 2, (LPARAM)pBuf);
    }
}

static char* stripPath(char* filename) {
    char* ptr = filename + strlen(filename) - 1;

    while (--ptr >= filename) {
        if (*ptr == '/' || *ptr == '\\') {
            return ptr + 1;
        }
    }

    return filename;
}

static void updateFileHistory(char* history, char* filename) {
    char fname[MAX_PATH];
    int i = 0;

    strcpy(fname, filename);

    for (i = 0; i < MAX_HISTORY - 1; i++) {
        if (*(history + MAX_PATH * i) == 0 || 0 == strcmp(history + MAX_PATH * i, fname)) {
            break;
        }
    }
    while (i > 0) {
        strcpy(history + MAX_PATH * i, history + MAX_PATH * (i - 1));
        i--;
    }
    strcpy(history, fname);
}

static void verifyFileHistory(char* history) {
    int i, j;

    for (i = 0; i < MAX_HISTORY; i++) {
        char *fname = history + MAX_PATH * i;
        if (fname[0] != '\0') {
            FILE* file = fopen(fname, "r");
            if (file != NULL) {
                fclose(file);
            }
            else {
                for (j = i + 1; j < MAX_HISTORY; j++) {
                    strcpy(history + MAX_PATH * (j - 1), history + MAX_PATH * j);
                    *(history + MAX_PATH * j) = 0;
                }
                i--;
            }
        }
    }
}

static HMENU createMenu() {
    HMENU hMenu    = CreateMenu();
    HMENU hFile    = CreateMenu();
    HMENU hCartA   = CreateMenu();
    HMENU hCartB   = CreateMenu();
    HMENU hDiskA   = CreateMenu();
    HMENU hDiskB   = CreateMenu();
    HMENU hRun     = CreateMenu();
    HMENU hWindow  = CreateMenu(); 
    HMENU hOptions = CreateMenu(); 
    HMENU hHelp    = CreateMenu();
    int i;

    /* FILE menu */

    /* Cartridge menus */
    AppendMenu(hCartA, MF_STRING, ID_FILE_INSERT_CARTRIDGEA, "Insert");
    AppendMenu(hCartA, MF_STRING | (*pProperties->cartridge.slotA ? 0 : MF_GRAYED), ID_FILE_REMOVE_CARTRIDGEA, "Remove");

    AppendMenu(hCartA, MF_SEPARATOR, 0, NULL);
    if (*pProperties->filehistory.cartridgeA[0] == 0) {
        AppendMenu(hCartA, MF_STRING | MF_GRAYED,  0, "- no recent files -");
    }
    verifyFileHistory(*pProperties->filehistory.cartridgeA);

    for (i = 0; i < MAX_HISTORY && *pProperties->filehistory.cartridgeA[i]; i++) {
        AppendMenu(hCartA, MF_STRING, ID_CARTRIDGEA_HISTORY + i, stripPath(pProperties->filehistory.cartridgeA[i]));
    }

    AppendMenu(hCartB, MF_STRING, ID_FILE_INSERT_CARTRIDGEB, "Insert");
    AppendMenu(hCartB, MF_STRING | (*pProperties->cartridge.slotB ? 0 : MF_GRAYED), ID_FILE_REMOVE_CARTRIDGEB, "Remove");

    AppendMenu(hCartB, MF_SEPARATOR, 0, NULL);
    if (*pProperties->filehistory.cartridgeB[0] == 0) {
        AppendMenu(hCartB, MF_STRING | MF_GRAYED,  0, "- no recent files -");
    }
    verifyFileHistory(*pProperties->filehistory.cartridgeB);
    for (i = 0; i < MAX_HISTORY && *pProperties->filehistory.cartridgeB[i]; i++) {
        AppendMenu(hCartB, MF_STRING, ID_CARTRIDGEB_HISTORY + i, stripPath(pProperties->filehistory.cartridgeB[i]));
    }

    /* Diskdrive menus */
    AppendMenu(hDiskA, MF_STRING, ID_FILE_INSERT_DISKETTEA, "Insert");
    AppendMenu(hDiskA, MF_STRING | (*pProperties->diskdrive.slotA ? 0 : MF_GRAYED), ID_FILE_REMOVE_DISKETTEA, "Eject");
    AppendMenu(hDiskA, MF_SEPARATOR, 0, NULL);
    AppendMenu(hDiskA, MF_STRING | (pProperties->diskdrive.autostartA ? MFS_CHECKED : 0), ID_FILE_INSERT_DISKETTEA_RESET, "Reset after insert");

    AppendMenu(hDiskA, MF_SEPARATOR, 0, NULL);
    if (*pProperties->filehistory.diskdriveA[0] == 0) {
        AppendMenu(hDiskA, MF_STRING | MF_GRAYED,  0, "- no recent files -");
    }
    verifyFileHistory(*pProperties->filehistory.diskdriveA);
    for (i = 0; i < MAX_HISTORY && *pProperties->filehistory.diskdriveA[i]; i++) {
        AppendMenu(hDiskA, MF_STRING, ID_DISKDRIVEA_HISTORY + i, stripPath(pProperties->filehistory.diskdriveA[i]));
    }

    AppendMenu(hDiskB, MF_STRING, ID_FILE_INSERT_DISKETTEB, "Insert");
    AppendMenu(hDiskB, MF_STRING | (*pProperties->cartridge.slotB ? 0 : MF_GRAYED), ID_FILE_REMOVE_DISKETTEB, "Eject");

    AppendMenu(hDiskB, MF_SEPARATOR, 0, NULL);
    if (*pProperties->filehistory.diskdriveB[0] == 0) {
        AppendMenu(hDiskB, MF_STRING | MF_GRAYED,  0, "- no recent files -");
    }
    verifyFileHistory(*pProperties->filehistory.diskdriveB);
    for (i = 0; i < MAX_HISTORY && *pProperties->filehistory.diskdriveB[i]; i++) {
        AppendMenu(hDiskB, MF_STRING, ID_DISKDRIVEB_HISTORY + i, stripPath(pProperties->filehistory.diskdriveB[i]));
    }

    /* Main file menu */
    AppendMenu(hFile, MF_POPUP,     (UINT)hCartA, "Cartridge slot 1");
    AppendMenu(hFile, MF_POPUP,     (UINT)hCartB, "Cartridge slot 2");
    AppendMenu(hFile, MF_SEPARATOR, 0, NULL);
    AppendMenu(hFile, MF_POPUP,     (UINT)hDiskA, "Disk drive A");
    AppendMenu(hFile, MF_POPUP,     (UINT)hDiskB, "Disk drive B");
    AppendMenu(hFile, MF_SEPARATOR, 0, NULL);
    AppendMenu(hFile, MF_STRING, ID_FILE_LOAD, "Load CPU state");
    AppendMenu(hFile, MF_STRING | (Prt.emuState != EMU_STOPPED ? 0 : MF_GRAYED), ID_FILE_SAVE, "Save CPU state");
    AppendMenu(hFile, MF_SEPARATOR, 0, NULL);
    AppendMenu(hFile, MF_STRING, ID_FILE_QLOAD, "Quick load state\tF7");
    AppendMenu(hFile, MF_STRING | (Prt.emuState != EMU_STOPPED ? 0 : MF_GRAYED), ID_FILE_QSAVE, "Quick save state\tF8");
    AppendMenu(hFile, MF_SEPARATOR, 0, NULL);
    AppendMenu(hFile, MF_STRING | (Prt.logSound ? MFS_CHECKED : 0), ID_LOG_WAV, "Log audio to file        \tF6");
    AppendMenu(hFile, MF_SEPARATOR, 0, NULL);
    AppendMenu(hFile, MF_STRING, ID_FILE_EXIT, "Exit");

    /* Run Menu */
    AppendMenu(hRun, MF_STRING, ID_RUN_RUN, Prt.emuState != EMU_RUNNING ? "Run\tF9" : "Pause\tF9");
    AppendMenu(hRun, MF_STRING | (Prt.emuState != EMU_STOPPED ? 0 : MF_GRAYED), ID_RUN_STOP, "Stop        \tF10");
    AppendMenu(hRun, MF_STRING, ID_RUN_RESET, "Reset\tF12");

    /* Window Menu */
    AppendMenu(hWindow, MF_STRING | (pProperties->video.size == P_VIDEO_SIZEX1 ? MFS_CHECKED : 0), ID_SIZE_NORMAL, "Normal size");
    AppendMenu(hWindow, MF_STRING | (pProperties->video.size == P_VIDEO_SIZEX2 ? MFS_CHECKED : 0), ID_SIZE_X2, "Double size");
    AppendMenu(hWindow, MF_STRING | (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN ? MFS_CHECKED : 0), ID_SIZE_FULLSCREEN, "Fullscreen");

    /* Options Menu */
    AppendMenu(hOptions, MF_STRING, ID_OPTIONS_EMULATION, "Emulation");
    AppendMenu(hOptions, MF_STRING, ID_OPTIONS_VIDEO, "Video");
    AppendMenu(hOptions, MF_STRING, ID_OPTIONS_AUDIO, "Sound");
    AppendMenu(hOptions, MF_STRING, ID_OPTIONS_CONTROLS, "Controls");

    /* Options Menu */
    AppendMenu(hHelp, MF_STRING, ID_HELP_ABOUT, "About");

    /* Main menu */
    AppendMenu(hMenu, MF_POPUP, (UINT)hFile, "File");
    AppendMenu(hMenu, MF_POPUP, (UINT)hRun, "Run");
    AppendMenu(hMenu, MF_POPUP, (UINT)hWindow, "Window");
    AppendMenu(hMenu, MF_POPUP, (UINT)hOptions, "Options");
    AppendMenu(hMenu, MF_POPUP, (UINT)hHelp, "Help");

    return hMenu;
}

static void showMenu(int showInFullscreen) 
{
    if (Prt.hMenu != NULL) {
        DestroyMenu(Prt.hMenu);
    }

    Prt.showMenu = showInFullscreen;
    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN && !showInFullscreen) {
        if (GetMenu(Prt.hwnd) != NULL) {
            SetMenu(Prt.hwnd, NULL);
            InvalidateRect(Prt.hwnd, NULL, 0);
            MouseEmulationEnable(Prt.hwnd);
        }
        return;
    }
    
    Prt.hMenu = createMenu();
    if (GetMenu(Prt.hwnd) != NULL) {
        SetMenu(Prt.hwnd, Prt.hMenu);
    }
    else {
        MouseEmulationDisable();
        SetMenu(Prt.hwnd, Prt.hMenu);
        InvalidateRect(Prt.hwnd, NULL, 0);
    }

    setStatusBar(-1, -1);
}

static void ScaleWindow(HWND hwnd) {
    int x, y, w, h;
    int zoom = getZoom();

    Prt.enteringFullscreen = 1;

    DirectXExitFullscreenMode(hwnd);

    if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
        ShowWindow(Prt.hwndStatus, FALSE);

        showMenu(0);

        SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_POPUP | WS_THICKFRAME | WS_DLGFRAME);

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            int depth = 32;
            switch (pProperties->video.fullRes) {
            case P_VIDEO_FRES320X240_16:
            case P_VIDEO_FRES640X480_16:
                depth = 16;
                break;
            case P_VIDEO_FRES320X240_32:
            case P_VIDEO_FRES640X480_32:
                depth = 32;
                break;
            }
            DirectXEnterFullscreenMode(hwnd, zoom * WIDTH, zoom * HEIGHT, depth);
        }

        x = -1 * GetSystemMetrics(SM_CXFIXEDFRAME) - 1;
        y = -1 * GetSystemMetrics(SM_CYFIXEDFRAME) - 1;
        w = GetSystemMetrics(SM_CXSCREEN) - 2 * x;
        h = GetSystemMetrics(SM_CYSCREEN) - 2 * y;

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            SetWindowPos(hwnd, 0, x, y, w, h, SWP_SHOWWINDOW | SWP_NOZORDER);
        }
        else {
            SetWindowPos(hwnd, HWND_TOPMOST, x, y, w, h, SWP_SHOWWINDOW); 
        }
    }
    else {
        if (GetWindowLong(hwnd, GWL_STYLE) & WS_POPUP) {
            showMenu(1);
            SetWindowLong(hwnd, GWL_STYLE, WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX);
            SetWindowPos(hwnd, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_SHOWWINDOW); 
        }

        w = WIDTH * zoom + Prt.DX;
        h = HEIGHT * zoom + Prt.DY;
        SetWindowPos(hwnd, NULL, Prt.X, Prt.Y, w, h, SWP_NOZORDER);
        
        ShowWindow(Prt.hwndStatus, TRUE);

        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            DirectXEnterWindowedMode(Prt.hwnd, zoom * WIDTH, zoom * HEIGHT);
        }
    }

    Prt.enteringFullscreen = 0;
}

void resetEmulator() 
{
    EmuState emuState = Prt.emuState;

    if (emuState == EMU_RUNNING) {
        Prt.emuState = EMU_PAUSED;
        soundSuspend();
    }
    Sleep(500);
    if (emuState != EMU_STOPPED) {
        char fileName[512];
        sprintf(fileName, "%s\\%s", Prt.pCurDir, "temp.sta");
        SaveState(fileName);
    }
    StopEmulator();
    if (emuState != EMU_STOPPED) {
        sprintf(Prt.pStoredState, "%s\\%s", Prt.pCurDir, "temp.sta");
    }

    StartEmulator();
    Prt.emuState = emuState;
    if (Prt.emuState != EMU_RUNNING) {
        soundSuspend();
    }
}

void updateVideoRender() {
    switch (pProperties->video.monType) {
    case P_VIDEO_COLOR:
        videoSetColorMode(Prt.pVideo, VIDEO_COLOR);
        break;
    case P_VIDEO_BW:
        videoSetColorMode(Prt.pVideo, VIDEO_BLACKWHITE);
        break;
    case P_VIDEO_GREEN:
        videoSetColorMode(Prt.pVideo, VIDEO_GREEN);
        break;
    }

    switch (pProperties->video.palEmu) {
    case P_VIDEO_PALNONE:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_FAST);
        break;
    case P_VIDEO_PALYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP);
        break;
    case P_VIDEO_PALNYC:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_SHARP_NOISE);
        break;
    case P_VIDEO_PALCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR);
        break;
    case P_VIDEO_PALNCOMP:
        videoSetPalMode(Prt.pVideo, VIDEO_PAL_BLUR_NOISE);
        break;
	case P_VIDEO_PALSCALE2X:
		videoSetPalMode(Prt.pVideo, VIDEO_PAL_SCALE2X);
		break;
    }
    videoSetFrameSkip(Prt.pVideo, pProperties->video.frameSkip);
}

void updateJoystick() {    
    switch (pProperties->joy1.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(1, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(1, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(1, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(1, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(1, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(1, JOY_HW2);
        break;
    }

    JoystickSetKeyStateKey(1, JOY_UP,    pProperties->joy1.keyUp);
    JoystickSetKeyStateKey(1, JOY_DOWN,  pProperties->joy1.keyDown);
    JoystickSetKeyStateKey(1, JOY_LEFT,  pProperties->joy1.keyLeft);
    JoystickSetKeyStateKey(1, JOY_RIGHT, pProperties->joy1.keyRight);
    JoystickSetKeyStateKey(1, JOY_BT1,   pProperties->joy1.button1);
    JoystickSetKeyStateKey(1, JOY_BT2,   pProperties->joy1.button2);

    JoystickSetKeyStateKey(2, JOY_UP,    pProperties->joy2.keyUp);
    JoystickSetKeyStateKey(2, JOY_DOWN,  pProperties->joy2.keyDown);
    JoystickSetKeyStateKey(2, JOY_LEFT,  pProperties->joy2.keyLeft);
    JoystickSetKeyStateKey(2, JOY_RIGHT, pProperties->joy2.keyRight);
    JoystickSetKeyStateKey(2, JOY_BT1,   pProperties->joy2.button1);
    JoystickSetKeyStateKey(2, JOY_BT2,   pProperties->joy2.button2);

    switch (pProperties->joy1.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(1, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(1, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(1, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(1, JOY_AF_FAST);
        break;
    }

    switch (pProperties->joy2.type) {
    case P_JOY_NONE:
    case P_JOY_MOUSE:
        JoystickSetType(2, JOY_NONE);
        break;
    case P_JOY_NUMPAD:
        JoystickSetType(2, JOY_NUMPAD);
        break;
    case P_JOY_KEYSETA:
        JoystickSetType(2, JOY_KEYSETA);
        break;
    case P_JOY_KEYSETB:
        JoystickSetType(2, JOY_KEYSETB);
        break;
    case P_JOY_HW1:
        JoystickSetType(2, JOY_HW1);
        break;
    case P_JOY_HW2:
        JoystickSetType(2, JOY_HW2);
        break;
    }

    switch (pProperties->joy2.autofire) {
    case P_JOY_AFOFF:
        JoystickSetAutofire(2, JOY_AF_OFF);
        break;
    case P_JOY_AFSLOW:
        JoystickSetAutofire(2, JOY_AF_SLOW);
        break;
    case P_JOY_AFMEDIUM:
        JoystickSetAutofire(2, JOY_AF_MEDIUM);
        break;
    case P_JOY_AFFAST:
        JoystickSetAutofire(2, JOY_AF_FAST);
        break;
    }
}

void showPropertiesDialog(int startPane) {
    Properties oldProp = *pProperties;
    int restart = 0;
	int i;

    int changed = showProperties(pProperties, Prt.hwnd, startPane, Prt.mixer);
    if (!changed) {
        return;
    }

    /* Save properties */
    propSave(pProperties);

    /* Always update video render */
    updateVideoRender();

    /* Always update joystick controls */
    updateJoystick();

    /* Update window size only if changed */
    if (pProperties->video.driver != oldProp.video.driver ||
        pProperties->video.fullRes != oldProp.video.fullRes ||
        pProperties->video.size != oldProp.video.size)
    {
        ScaleWindow(Prt.hwnd);
    }

    /* Must restart if video type changed */
    restart |= pProperties->video.videoType != oldProp.video.videoType;

    /* Must restart MSX if joystick type changed between joystick, mouse or none */
    restart |= oldProp.joy1.type == P_JOY_NONE  && pProperties->joy1.type != P_JOY_NONE;
    restart |= oldProp.joy1.type != P_JOY_NONE  && pProperties->joy1.type == P_JOY_NONE;
    restart |= oldProp.joy1.type == P_JOY_MOUSE && pProperties->joy1.type != P_JOY_MOUSE;
    restart |= oldProp.joy1.type != P_JOY_MOUSE && pProperties->joy1.type == P_JOY_MOUSE;
    restart |= oldProp.joy2.type == P_JOY_NONE  && pProperties->joy2.type != P_JOY_NONE;
    restart |= oldProp.joy2.type != P_JOY_NONE  && pProperties->joy2.type == P_JOY_NONE;
    restart |= oldProp.joy2.type == P_JOY_MOUSE && pProperties->joy2.type != P_JOY_MOUSE;
    restart |= oldProp.joy2.type != P_JOY_MOUSE && pProperties->joy2.type == P_JOY_MOUSE;

    /* Must restart MSX if CPU or ram configuration changed */
    if (memcmp(&oldProp.emulation, &pProperties->emulation, sizeof(oldProp.emulation))) {
        restart = 1;
    }

    /* Update sound only if changed, Must restart if changed */
    //if (memcmp(&oldProp.sound, &pProperties->sound, sizeof(oldProp.sound))) {
    if (oldProp.sound.bufSize   != pProperties->sound.bufSize ||
        oldProp.sound.driver    != pProperties->sound.driver  ||
        oldProp.sound.frequency != pProperties->sound.frequency ||
        oldProp.sound.stereo    != pProperties->sound.stereo) 
    {

        soundDestroy();
        soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), pProperties->sound.stereo ? 2 : 1);
        restart = 1;
    }

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }

    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    if (restart) {
        resetEmulator();
    }

    showMenu(0);

    InvalidateRect(Prt.hwnd, NULL, TRUE);
}


static void insertCartridge(int drive, char* filename) {
    StopEmulator();
    if (drive == 0) {
        strcpy(pProperties->cartridge.slotA, filename);
        updateFileHistory(*pProperties->filehistory.cartridgeA, filename);
    }
    else {
        strcpy(pProperties->cartridge.slotB, filename);
        updateFileHistory(*pProperties->filehistory.cartridgeB, filename);
    }
    StartEmulator();
}

static void insertDiskette(int drive, char* filename) {
    if (drive == 0) {
        if (pProperties->diskdrive.autostartA) {
            StopEmulator();
        }
        strcpy(pProperties->diskdrive.slotA, filename);
        updateFileHistory(*pProperties->filehistory.diskdriveA, filename);
        if (pProperties->diskdrive.autostartA) {
            StartEmulator();
        }
        else if (Prt.emuState != EMU_STOPPED) {
            ChangeDisk(0, filename);
        }
    }
    else {
        strcpy(pProperties->diskdrive.slotB, filename);
        updateFileHistory(*pProperties->filehistory.diskdriveB, filename);
        if (Prt.emuState != EMU_STOPPED) {
            ChangeDisk(1, filename);
        }
    }
}

static LRESULT CALLBACK wndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam) 
{
    PAINTSTRUCT ps;
    HDC hdc;
    RECT r;
    int i;
    char* pFname;

    switch (iMsg) {
    case WM_CREATE:
        DragAcceptFiles(hwnd, TRUE);
        return 0;

    case WM_DROPFILES:
        {
            char fname[MAX_PATH];
            int len;
            HDROP hDrop;
            hDrop = (HDROP)wParam;
            DragQueryFile(hDrop, 0, fname, 512);
            DragFinish (hDrop);
            len = strlen(fname);
            if (len > 4) {
                if (toupper(fname[len - 4]) == '.' &&
                    toupper(fname[len - 3]) == 'R' &&
                    toupper(fname[len - 2]) == 'O' &&
                    toupper(fname[len - 1]) == 'M')
                {
                    insertCartridge(0, fname);
                }
                else 
                if (toupper(fname[len - 4]) == '.' &&
                    toupper(fname[len - 3]) == 'D' &&
                    toupper(fname[len - 2]) == 'S' &&
                    toupper(fname[len - 1]) == 'K')
                {
                    insertDiskette(0, fname);
                }
                showMenu(0);
            }
        }
        return 0;

    case WM_COMMAND:
        i = LOWORD(wParam) - ID_CARTRIDGEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertCartridge(0, pProperties->filehistory.cartridgeA[i]);
        }
        i = LOWORD(wParam) - ID_CARTRIDGEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertCartridge(1, pProperties->filehistory.cartridgeB[i]);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEA_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertDiskette(0, pProperties->filehistory.diskdriveA[i]);
        }
        i = LOWORD(wParam) - ID_DISKDRIVEB_HISTORY;
        if (i >= 0 && i < MAX_HISTORY) {
            insertDiskette(1, pProperties->filehistory.diskdriveB[i]);
        }
        switch (LOWORD(wParam)) {
        case ID_LOG_WAV:
            if (Prt.logSound) {
                mixerStopLog(Prt.mixer);
                Prt.logSound = 0;
            }
            else {
                char fileName[512];
                sprintf(fileName, "%s\\msx%03d.wav", Prt.pCurDir, pProperties->sound.logFileCount++);
                Prt.logSound = mixerStartLog(Prt.mixer, fileName);
            }
            break;
        case ID_FILE_SAVE:
            if (Prt.emuState != EMU_STOPPED) {
                suspendEmulator();
                pFname = saveFile(hwnd, "Save CPU state", "CPU state   (*.sta)\0*.sta\0", NULL);
                if (pFname != NULL) {
                    SaveState(pFname);
                    free(pFname);
                }
                resumeEmulator();
            }
            break;
        case ID_FILE_LOAD:
            suspendEmulator();
            pFname = openFile(hwnd, "Load CPU state", "CPU state   (*.sta)\0*.sta\0", NULL);
            if (pFname != NULL) {
                StopEmulator();
                strcpy(Prt.pStoredState, pFname);
                free(pFname);
                StartEmulator();
            }
            else {
                resumeEmulator();
            }
            break;
        case ID_FILE_QSAVE:
            if (Prt.emuState != EMU_STOPPED) {
                char fileName[512];
                suspendEmulator();
                sprintf(fileName, "%s\\%s", Prt.pCurDir, "quicksave.sta");
                SaveState(fileName);
                resumeEmulator();
            }
            break;
        case ID_FILE_QLOAD:
            StopEmulator();
            sprintf(Prt.pStoredState, "%s\\%s", Prt.pCurDir, "quicksave.sta");
            StartEmulator();
            break;
        case ID_FILE_INSERT_CARTRIDGEA:
            suspendEmulator();
            pFname = openFile(hwnd, "Insert ROM cartridge", "ROM cartridge   (*.rom, *.zip)\0*.rom; *.zip\0", NULL);
            if (pFname != NULL) {
                insertCartridge(0, pFname);
            }
            else {
                resumeEmulator();
            }
            break;
        case ID_FILE_INSERT_CARTRIDGEB:
            suspendEmulator();
            pFname = openFile(hwnd, "Insert ROM cartridge", "ROM cartridge   (*.rom, *.zip)\0*.rom; *.zip\0", NULL);
            if (pFname != NULL) {
                insertCartridge(1, pFname);
            }
            else {
                resumeEmulator();
            }
            break;
        case ID_FILE_INSERT_DISKETTEA_RESET:
            pProperties->diskdrive.autostartA ^= 1;
            break;
        case ID_FILE_INSERT_DISKETTEA:
            suspendEmulator();
            pFname = openFile(hwnd, "Insert disk image", "Disk image   (*.dsk, *.zip)\0*.dsk; *.zip\0", NULL);
            if (pFname != NULL) {
                insertDiskette(0, pFname);
            }
            resumeEmulator();
            break;
        case ID_FILE_INSERT_DISKETTEB:
            suspendEmulator();
            pFname = openFile(hwnd, "Insert disk image", "Disk image   (*.dsk, *.zip)\0*.dsk; *.zip\0", NULL);
            if (pFname != NULL) {
                insertDiskette(1, pFname);
            }
            resumeEmulator();
            break;
        case ID_FILE_REMOVE_CARTRIDGEA:
            pProperties->cartridge.slotA[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                StopEmulator();
                StartEmulator();
            }
            break;
        case ID_FILE_REMOVE_CARTRIDGEB:
            pProperties->cartridge.slotB[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                StopEmulator();
                StartEmulator();
            }
            break;
        case ID_FILE_REMOVE_DISKETTEA:
            pProperties->diskdrive.slotA[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                ChangeDisk(0, NULL);
            }
            break;
        case ID_FILE_REMOVE_DISKETTEB:
            pProperties->diskdrive.slotB[0] = 0;
            if (Prt.emuState != EMU_STOPPED) {
                ChangeDisk(1, NULL);
            }
            break;
        case ID_FILE_EXIT:
            DestroyWindow(hwnd);
            break;
        case ID_SIZE_NORMAL:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX1)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_X2:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEX2)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_SIZE_FULLSCREEN:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
            }
            break;
        case ID_RUN_RUN:
            if (Prt.emuState == EMU_STOPPED) {
                StartEmulator();
            }
            else if (Prt.emuState == EMU_PAUSED) {
                soundResume();
                Prt.emuState = EMU_RUNNING;
            }
            else {
                soundSuspend();
                Prt.emuState = EMU_PAUSED;
            }
            break;
        case ID_RUN_STOP:
            if (Prt.emuState != EMU_STOPPED) {
                StopEmulator();
                InvalidateRect(Prt.hwnd, NULL, TRUE);
            }
            break;
        case ID_RUN_RESET:
            StopEmulator();
            StartEmulator();
            break;
        case ID_OPTIONS_EMULATION:
            showPropertiesDialog(PROP_EMULATION);
            break;
        case ID_OPTIONS_VIDEO:
            showPropertiesDialog(PROP_VIDEO);
            break;
        case ID_OPTIONS_AUDIO:
            showPropertiesDialog(PROP_SOUND);
            break;
        case ID_OPTIONS_CONTROLS:
            showPropertiesDialog(PROP_CONTROLS);
            break;
        case ID_HELP_ABOUT:
            {
                char info[2048];
                //sprintf (info, "%s%s%s%d%s",
				sprintf (info, "%s%s%s",
                    "        RedMSX version ", RED_MSX_VERSION, "\n\n"
                    "        Developed by SLotman\n\n"
                    "        The program is based on blueMSX 1.0.5 by Daniel Vik.\n"
                    "        Ported to Visual C++ 6, added rom and dsk zip loading,\n"
                    "        Scale2X and screenshot support by SLotman\n"
                    "        ---------------------------------------------------------------------------------------\n"
                    "        Thanks to Daniel Caetano, Felipe Bergo and Leandro                \n"
                    "        Pereira for all the help, to Julio Marchi for helping me out,\n"
                    "        and to Daniel Vik for his great emulator.\n"
                    "        ---------------------------------------------------------------------------------------\n"
                    "        Visit http://www.msxfiles.tk\n"
					);

                MessageBox(NULL, info, "RedMSX - About", MB_ICONINFORMATION | MB_OK);
            }
            break;
        }

        showMenu(0);
        break;

    case WM_SYSKEYUP:
    case WM_SYSKEYDOWN:
    case WM_KEYUP:
    case WM_KEYDOWN:
        switch (wParam) {
        case VK_F6:
            if (iMsg == WM_KEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_LOG_WAV, 0);
            }
            return 0;
			break;
        case VK_F7:
            if (iMsg == WM_KEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QLOAD, 0);
            }
            return 0;
			break;
        case VK_F8:
            if (iMsg == WM_KEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_FILE_QSAVE, 0);
            }
            return 0;
			break;
        case VK_F9:
            if (iMsg == WM_KEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RUN, 0);
            }
            return 0;
			break;
        case VK_F10:
            if (iMsg == WM_SYSKEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_STOP, 0);
            }
            return 0;
			break;
        case VK_F12:
            if (iMsg == WM_KEYUP) {
                SendMessage(hwnd, WM_COMMAND, ID_RUN_RESET, 0);
            }
            return 0;
			break;
		case VK_SNAPSHOT:
			{
				ScreenShot(hwnd);
				return 0;
			}
			break;
        case VK_APPS:
            if (iMsg == WM_KEYUP) {
                showMenu(1);
            }
            return 0;
			break;
        }
		UpdateKeyboardState();

        return 0;

    case WM_SYSCOMMAND:
        switch(wParam) {
        case SC_MAXIMIZE:
            if (propSetVideoSize(pProperties, P_VIDEO_SIZEFULLSCREEN)) {
                ScaleWindow(hwnd);
                showMenu(0);
            }
            return 0;
        case SC_MINIMIZE:
            suspendEmulator();
            break;
        case SC_RESTORE:
            resumeEmulator();
            break;
        }
        break;

    case WM_ENTERSIZEMOVE:
        suspendEmulator();
        break;

    case WM_EXITSIZEMOVE:
        resumeEmulator();
        break;

    case WM_ENTERMENULOOP:
        suspendEmulator();
        return 0;

    case WM_EXITMENULOOP:
        resumeEmulator();
        showMenu(0);
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            PostMessage(hwnd, WM_LBUTTONDOWN, 0, 0);
            PostMessage(hwnd, WM_LBUTTONUP, 0, 0);
        }
        return 0;


    case WM_MOVE:
        if (!Prt.enteringFullscreen && pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }

    case WM_DISPLAYCHANGE:
        if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
            int zoom = getZoom();
            if (Prt.enteringFullscreen) {
                DirectXUpdateWindowedMode(hwnd, zoom * WIDTH, zoom * HEIGHT);
            }
        }
        break;

    case WM_SIZE:
        MoveWindow(Prt.hwndStatus, 0, 0, 0, 0, TRUE);
        InvalidateRect(hwnd, NULL, FALSE);
        break;
        
    case WM_ACTIVATE:
        if (wParam == WA_INACTIVE) {
            MouseEmulationDisable();
        }
        else {
            MouseEmulationEnable(hwnd);
        }
        break;

    case WM_MOUSEMOVE:
        if (pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
            if (HIWORD(lParam) < 2) {
                if (!Prt.showMenu) {
                    Prt.showMenu = 1;
                    Sleep(200);
                }
                showMenu(1);
            }
        }
        break;

    case WM_LBUTTONDOWN:
		showMenu(0);
        break;

    case WM_ERASEBKGND:
        return 1;

    case WM_UPDATESTATUSBAR:
        setStatusBar(wParam, lParam);
        return 0;

    case WM_UPDATE:
        InvalidateRect(hwnd, NULL, FALSE);
        return 0;

    case WM_INPUTLANGCHANGE:
        InitKeyboard();
        break;

    case WM_PAINT:
        {
            int dy = 0;
            BITMAP bm;
            HDC hMemDC;
            HBITMAP hBitmap;
			int srcDoubleWidth = 0;

			if (Prt.screenMode == 6 || Prt.screenMode == 7 || Prt.screenMode == 13) {
                srcDoubleWidth = 1;
            }

            hdc = BeginPaint(hwnd, &ps);   

            GetClientRect(hwnd, &r);
            if (Prt.showMenu && pProperties->video.size == P_VIDEO_SIZEFULLSCREEN) {
                int cy = GetSystemMetrics(SM_CYSCREEN);
                dy = r.bottom - cy;
                r.bottom = cy;
            }

            if (Prt.emuState == EMU_STOPPED) {
                hMemDC = CreateCompatibleDC(hdc);
                hBitmap = (HBITMAP)SelectObject(hMemDC, Prt.hLogo);
                GetObject(Prt.hLogo, sizeof(BITMAP), (PSTR)&bm);
                {
                    int x = ((r.right - r.left) - bm.bmWidth) / 2;
                    int y = ((r.bottom - r.top) - bm.bmHeight) / 2 + dy;

                    if (x > 0 || y > 0) {
                        StretchBlt(hdc, 0, dy, r.right - r.left, r.bottom - r.top, hMemDC, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
                    }
                    else {
                        BitBlt(hdc, x, y, bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
                    }
                }
                SelectObject(hMemDC, hBitmap);
                DeleteDC(hMemDC);
            }
            else {
                int width = r.right - r.left;
                int height = r.bottom - r.top - (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? Prt.DDY : 0);
                int zoom = getZoom();

                if (pProperties->video.driver == P_VIDEO_DRVDIRECTX) {
                    if (Prt.emuState != EMU_STOPPED) {
                        DirectXUpdateSurface(Prt.pVideo, Prt.bmBits, WIDTH, HEIGHT, srcDoubleWidth, dy, zoom);
                    }
                }
                else {
                    DWORD* bmBitsSrc = (DWORD*)Prt.bmBits + WIDTH * (HEIGHT - 1) * (srcDoubleWidth ? 2 : 1);
                    
                    videoRender(Prt.pVideo, 32, zoom, bmBitsSrc, WIDTH, HEIGHT, srcDoubleWidth, Prt.bmBitsGDI, 
                                -1 * (int)sizeof(DWORD) * WIDTH, zoom * WIDTH * sizeof(DWORD));
                    Prt.bmInfo.bmiHeader.biWidth    = zoom * WIDTH;
                    Prt.bmInfo.bmiHeader.biHeight   = zoom * HEIGHT;
                    Prt.bmInfo.bmiHeader.biBitCount = 32;
                    StretchDIBits(hdc, 0, dy + (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN ? 1 : 0), 
                                  width, height, 0, 0, zoom * WIDTH, zoom * HEIGHT, Prt.bmBitsGDI, 
                                  &Prt.bmInfo, DIB_RGB_COLORS, SRCCOPY);
                }
            }
            EndPaint(hwnd, &ps);
            Prt.updatePending = 0;
        }
        return 0;
        
    case WM_DESTROY:         
        if (pProperties->video.size != P_VIDEO_SIZEFULLSCREEN) {
            RECT r;
            
            GetWindowRect(hwnd, &r);
            Prt.X = r.left;
            Prt.Y = r.top;
        }
        Prt.enteringFullscreen = 1;
        DirectXExitFullscreenMode(hwnd);
        PostQuitMessage(0);
        StopEmulator();
        Sleep(400);
        return 0;
    }

    return DefWindowProc(hwnd, iMsg, wParam, lParam);
}

#ifdef _CONSOLE
int main(int argc, char **argv)
#else
WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, PSTR szLine, int iShow)
#endif
{
    static WNDCLASSEX wndClass;
    HINSTANCE hInstance = (HINSTANCE)GetModuleHandle(NULL);
    RECT wr;
    RECT cr;
    RECT sr;
    MSG msg;
	int i;

	WIN32_FIND_DATA FileData;
    HANDLE hFile;
	char lastfile[256];
	char filenum[4];

    InitCommonControls(); 
	InitKeyboard();

    wndClass.cbSize         = sizeof(wndClass);
    wndClass.style          = CS_OWNDC;
    wndClass.lpfnWndProc    = wndProc;
    wndClass.cbClsExtra     = 0;
    wndClass.cbWndExtra     = 0;
    wndClass.hInstance      = hInstance;
    wndClass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON9));
    wndClass.hIconSm        = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON9));
    wndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground  = NULL;
    wndClass.lpszMenuName   = NULL;
    wndClass.lpszClassName  = "RedMSX";

    RegisterClassEx(&wndClass);

    pProperties = propCreate();

    memset(Prt.keyMap, 0xff, 16);

    Prt.enteringFullscreen = 1;
    Prt.hMenu = createMenu();
    Prt.emuState = EMU_STOPPED;
    Prt.hLogo = LoadBitmap(hInstance, "LOGO");
    Prt.updatePending = 0;
    Prt.pVideo = videoCreate();
    Prt.mixer  = mixerCreate();

    /* Set SCREEN8 colors */
    for (i=0; i<256; i++) {
        BPal[i] = videoGetColor(Prt.pVideo, (i & 0x1c) << 3, (i & 0xe0), ((i & 3) == 3 ? 7 : 2 * (i & 3)) << 5);
    }

    Prt.X = CW_USEDEFAULT;
    Prt.Y = CW_USEDEFAULT;

    GetCurrentDirectory(MAX_PATH - 1, Prt.pCurDir);

	// screenshot ammount
	ttshots=0;
	mkdir("shots");
    hFile = FindFirstFile("shots\\shot????.bmp",&FileData);
    if (hFile==INVALID_HANDLE_VALUE) 
	{
       ttshots=0;
    } 
	else 
	{
		strcpy(lastfile,FileData.cFileName);
	    while (FindNextFile(hFile, &FileData )!=0)
		{
		   strcpy(lastfile,FileData.cFileName);
		}
		// now make some sense out of it
		filenum[0]=lastfile[4];
		filenum[1]=lastfile[5];
		filenum[2]=lastfile[6];
		filenum[3]=lastfile[7];
		filenum[4]='\0';
		ttshots=atoi(filenum);
    }

    Prt.hwnd = CreateWindow("RedMSX", "  RedMSX", 
                            WS_OVERLAPPED | WS_CLIPCHILDREN | WS_BORDER | WS_DLGFRAME | 
                            WS_SYSMENU | WS_MINIMIZEBOX | WS_MAXIMIZEBOX, 
                            CW_USEDEFAULT, CW_USEDEFAULT, 200, 200, NULL, Prt.hMenu, hInstance, NULL);


    Prt.hwndStatus = CreateWindowEx(0, STATUSCLASSNAME, NULL, WS_CHILD | WS_VISIBLE, 
                                    0, 0, 50, 50, Prt.hwnd, (HMENU) 443, hInstance, NULL);

    setStatusBar(-1, -1);

    GetWindowRect(Prt.hwnd, &wr);
    GetClientRect(Prt.hwnd, &cr);
    GetWindowRect(Prt.hwndStatus, &sr);

    Prt.X  = wr.left;
    Prt.Y  = wr.top;
    Prt.DY = (wr.bottom - wr.top) - (cr.bottom - cr.top);
    Prt.DDY = (sr.bottom - sr.top) - 1;
    Prt.DX = (wr.right - wr.left) - (cr.right - cr.left);

    SetWindowPos(Prt.hwnd, NULL, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOZORDER);

    Prt.enteringFullscreen = 0;

    Prt.bmInfo.bmiHeader.biSize           = sizeof(BITMAPINFOHEADER);
    Prt.bmInfo.bmiHeader.biWidth          = WIDTH;
    Prt.bmInfo.bmiHeader.biHeight         = HEIGHT;
    Prt.bmInfo.bmiHeader.biPlanes         = 1;
    Prt.bmInfo.bmiHeader.biBitCount       = 32;
    Prt.bmInfo.bmiHeader.biCompression    = BI_RGB;
    Prt.bmInfo.bmiHeader.biClrUsed        = 0;
    Prt.bmInfo.bmiHeader.biClrImportant   = 0;

    //Prt.bmBits    = calloc(WIDTH * HEIGHT, sizeof(pixel));
    Prt.bmBits    = calloc(2 * WIDTH * HEIGHT, sizeof(pixel));
    Prt.bmBitsGDI = malloc(4096 * 4096 * sizeof(pixel));
    Prt.logSound = 0;

	soundCreate(Prt.hwnd, pProperties->sound.driver, Prt.mixer, 44100, 50 * (pProperties->sound.bufSize + 2), pProperties->sound.stereo ? 2 : 1);

    for (i = 0; i < MIXER_CHANNEL_COUNT; i++) {
        mixerSetChannelVolume(Prt.mixer, i, pProperties->sound.mixerChannel[i].volume);
        mixerSetChannelPan(Prt.mixer, i, pProperties->sound.mixerChannel[i].pan);
        mixerEnableChannel(Prt.mixer, i, pProperties->sound.mixerChannel[i].enable);
    }
    
    mixerSetMasterVolume(Prt.mixer, pProperties->sound.masterVolume);

    updateVideoRender();
    updateJoystick();
    showMenu(0);
    ScaleWindow(Prt.hwnd);

    ShowWindow(Prt.hwnd, TRUE);
    UpdateWindow(Prt.hwnd);

    setStatusBar(-1, -1);

    while (GetMessage(&msg, NULL, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
    
    Sleep(500);
    videoDestroy(Prt.pVideo);
    propDestroy(pProperties);

    soundDestroy();
    Sleep(500);
    mixerDestroy(Prt.mixer);


    exit(0);

    return 0;
}

void StartEmulator() {
    DWORD id;
    ExitNow = 0;
    Prt.hThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)ThreadProc, 0, 0, &id);
    
    soundResume();

    Prt.emuState = EMU_RUNNING;
}

void StopEmulator() {
    if (Prt.emuState == EMU_STOPPED) {
        return;
    }

    Prt.emuState = EMU_STOPPED;
    ExitNow = 1;
    ReleaseSemaphore(Prt.hEvent, 100, NULL);
    WaitForSingleObject(Prt.hThread, INFINITE);
    Prt.pStoredState[0] = 0;
    soundSuspend();
    CloseHandle(Prt.hThread);
    CloseHandle(Prt.hEvent);
}

DWORD WINAPI ThreadProc(void* param) 
{
    int success;

    ResetMSX();

    ProgDir    = Prt.pCurDir;
    XBuf       = (char*)Prt.bmBits;
    SyncPeriod = 3580 * getSyncPeriod();
    UPeriod    = 1;
    VPeriod    = pProperties->video.videoType == P_VIDEO_PAL ? CPU_VPERIOD_PAL : CPU_VPERIOD_NTSC;
    CartA      = strlen(pProperties->cartridge.slotA) ? pProperties->cartridge.slotA : "CARTA.ROM";
    CartB      = strlen(pProperties->cartridge.slotB) ? pProperties->cartridge.slotB : "CARTB.ROM";
    //CartAZip   = strlen(pProperties->cartridge.slotAZip) ? pProperties->cartridge.slotAZip : NULL;
    //CartBZip   = strlen(pProperties->cartridge.slotBZip) ? pProperties->cartridge.slotBZip : NULL;
    DiskA      = strlen(pProperties->diskdrive.slotA) ? pProperties->diskdrive.slotA : "DRIVEA.DSK";
    DiskB      = strlen(pProperties->diskdrive.slotB) ? pProperties->diskdrive.slotB : "DRIVEB.DSK";
    //DiskAZip   = strlen(pProperties->diskdrive.slotAZip) ? pProperties->diskdrive.slotAZip : NULL;
    //DiskBZip   = strlen(pProperties->diskdrive.slotBZip) ? pProperties->diskdrive.slotBZip : NULL;
    CasName    = "DEFAULT.CAS";
    JoyTypeA   = pProperties->joy1.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy1.type == P_JOY_MOUSE ? 3 : 1;
    JoyTypeB   = pProperties->joy2.type == P_JOY_NONE  ? 0 : 
                 pProperties->joy2.type == P_JOY_MOUSE ? 3 : 1;
    StateName  = strlen(Prt.pStoredState) ? Prt.pStoredState : NULL;
    switch (pProperties->emulation.family) {
    case P_EMU_MSX:
        MSXVersion   = 0;
        MSXSysRom    = romMSX;
        MSXSysRomExt = NULL;
        break;
    case P_EMU_MSX2:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2;
        MSXSysRomExt = romMSX2EXT;
        break;
    case P_EMU_MSX2P:
        MSXVersion   = 2;
        MSXSysRom    = romMSX2P;
        MSXSysRomExt = romMSX2PEXT;
        break;
    case P_EMU_MSXBR:
        MSXVersion = 0;
        MSXSysRom  = romMSXBR;
        MSXSysRomExt = NULL;
        break;
    case P_EMU_MSXJ:
        MSXVersion   = 0;
        MSXSysRom    = romMSXJ;
        MSXSysRomExt = NULL;
        break;
    case P_EMU_MSX2J:
        MSXVersion   = 1;
        MSXSysRom    = romMSX2J;
        MSXSysRomExt = romMSX2EXTJ;
        break;
    }
    RAMPages   = 1 << pProperties->emulation.RAMsize;
    VRAMPages  = 1 << pProperties->emulation.VRAMsize;
    ComName    = NULL;
    PrnName    = NULL;
    TheMixer   = Prt.mixer;

    Prt.hEvent  = CreateSemaphore(NULL, 0, 1000, NULL);
    Prt.timerId = StartTimer(Prt.hwnd, Prt.hEvent, getSyncPeriod(), pProperties->emulation.syncMethod == P_EMU_SYNC1MS);

    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_ABOVE_NORMAL);

    success = StartMSX();
    TrashMSX();

    StopTimer(Prt.timerId);
    if (!success) {
        MessageBox(NULL, "Failed to Start MSX emulator.", "blueMSX - Error", MB_ICONHAND | MB_OK);
        PostMessage(Prt.hwnd, WM_COMMAND, ID_RUN_STOP, 0);
    }

    ExitThread(0);
    
    return 0;
}

void SetColor(register byte N,register byte R,register byte G,register byte B)
{
    DWORD J = videoGetColor(Prt.pVideo, R, G, B);
    if (N) {
        XPal[N] = J; 
    }
    else {
        XPal0 = J;
    }
}


byte Joystick(register byte N) {
    return JoystickGetState(N + 1);
}

void Mouse2(int* dx, int* dy, int* buttons) {
    MouseEmulationGetState(dx, dy, buttons);
}

void Keyboard(void)
{
    memcpy(KeyMap, Prt.keyMap, 16);
}

void WaitForSync(void) 
{
    do {
        suspendFlag = 1;
        WaitForSingleObject(Prt.hEvent, INFINITE);
    } while (!ExitNow && Prt.emuState != EMU_RUNNING);
    suspendFlag = 0;

}

void RefreshScreen(int screenMode)
{
    static int frameSkip = 0;
    if (frameSkip++ < pProperties->video.frameSkip) {
        return;
    }
    frameSkip = 0;

    Prt.updatePending++;
    if (Prt.updatePending <= 1) {
        Prt.screenMode = screenMode;
        PostMessage(Prt.hwnd, WM_UPDATE, screenMode, 0);
    }
}
/*
void PutImage(void)
{
    static int frameSkip = 0;
    if (frameSkip++ < pProperties->video.frameSkip) {
        return;
    }
    frameSkip = 0;

    Prt.updatePending++;
    if (Prt.updatePending <= 1) {
        PostMessage(Prt.hwnd, WM_UPDATE, 0, 0);
    }
}
*/

HRESULT SaveBitmap(char strFileName[128], PBITMAPINFO pbi, HBITMAP hBMP, HDC hDC)
{
	HRESULT				hr = E_FAIL;
	HANDLE				hf;			// file handle
	BITMAPFILEHEADER	hdr;		// bitmap file-header
	PBITMAPINFOHEADER	pbih;		// bitmap info-header
	LPBYTE				lpBits;		// memorypointer
	DWORD				dwTotal;	// total count of bytes
	DWORD				cb;			// incremental count of bytes
	BYTE				*hp;		// byte pointer
	DWORD				dwTmp;		// temp-variable


	if( pbi == NULL ){
		return E_FAIL;
	}

	pbih	= ( PBITMAPINFOHEADER ) pbi;
	lpBits	= ( LPBYTE ) GlobalAlloc( GMEM_FIXED, pbih->biSizeImage );

	if( !lpBits ){
		return E_FAIL;
	}


	//
	// Retrieve the colortable ( RGBQUAD-array ) and the bits
	// ( array of palette indices ) from the DIB.
	//
	if( !GetDIBits( hDC, hBMP, 0, ( WORD )pbih->biHeight, lpBits, pbi, DIB_RGB_COLORS ) ){
		return E_FAIL;
	}


	//
	// Create the .BMP file.
	//
	hf = CreateFile( strFileName, GENERIC_READ|GENERIC_WRITE, ( DWORD ) 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, ( HANDLE ) NULL );

	if( hf == INVALID_HANDLE_VALUE ){
		return E_FAIL;
	}

	hdr.bfType = 0x4D42;	// 0x42 = "B", 0x4D = "M"


	//
	// Compute the size of the entire file
	//
	hdr.bfSize = ( DWORD )( sizeof( BITMAPFILEHEADER ) + pbih->biSize + pbih->biClrUsed * sizeof( RGBQUAD ) + pbih->biSizeImage );
	hdr.bfReserved1 = 0;
	hdr.bfReserved2 = 0;

	//
	// Compute the offset to the array of color indices
	//
	hdr.bfOffBits = ( DWORD ) sizeof( BITMAPFILEHEADER ) + pbih->biSize + pbih->biClrUsed * sizeof( RGBQUAD );


	//
	// Copy the BITMAPFILEHEADER into the .BMP file
	//
	if( !WriteFile( hf, ( LPVOID ) &hdr, sizeof( BITMAPFILEHEADER ), ( LPDWORD ) &dwTmp, NULL ) ){
		return E_FAIL;
	}


	//
	// Copy the BITMAPINFOHEADER and RGBQUAD array into the file
	//
	if( !WriteFile( hf, ( LPVOID ) pbih, sizeof( BITMAPINFOHEADER ) + pbih->biClrUsed * sizeof( RGBQUAD ), ( LPDWORD ) &dwTmp, ( NULL ) ) ){
		return E_FAIL;
	}


	//
	// Copy the array of color indices into the .BMP file
	//
	dwTotal = cb = pbih->biSizeImage;
	hp      = lpBits;

	if( !WriteFile( hf, ( LPSTR ) hp, ( int ) cb, ( LPDWORD ) &dwTmp, NULL ) ){
		return E_FAIL;
	}


	//
	// Now close the .BMP file
	//
	if( !CloseHandle( hf ) ){
		return E_FAIL;
	}


	//
	// Free the memory reserved
	//
	GlobalFree( ( HGLOBAL ) lpBits );


	return S_OK;
}

PBITMAPINFO CreateBitmapInfoStructure(HBITMAP hBmp)
{
	BITMAP	    bmp;
	PBITMAPINFO pbmi;
	WORD		cClrBits;


	//
	// Get the width, height and the colordepth of the image
	//
	if( !GetObject( hBmp, sizeof( BITMAP ), (LPSTR)&bmp ) ){
		return NULL;
	}


	//
	// Convert the color format to a count of bits
	//
	cClrBits = 24;


	//
	// Allocate memory for the BITMAPINFO-structure. This structure
	// contains a BITMAPINFOHEADER-structure and an array of RGBQUAD
	// datastructures.
	//
	if( cClrBits < 24 )
		pbmi = ( PBITMAPINFO ) LocalAlloc( LPTR, sizeof( BITMAPINFOHEADER ) + sizeof( RGBQUAD ) * ( 1 << cClrBits ) );
	else
		pbmi = ( PBITMAPINFO ) LocalAlloc( LPTR, sizeof( BITMAPINFOHEADER ) );

	//
	// Initialize the fields in the BITMAPINFO-structure.
	//
	pbmi->bmiHeader.biSize		= sizeof( BITMAPINFOHEADER );
	pbmi->bmiHeader.biWidth		= bmp.bmWidth;
	pbmi->bmiHeader.biHeight	= bmp.bmHeight;
	pbmi->bmiHeader.biPlanes	= bmp.bmPlanes;
	pbmi->bmiHeader.biBitCount	= cClrBits;//bmp.bmBitsPixel;

	if( cClrBits < 24 )
		pbmi->bmiHeader.biClrUsed = ( 1 << cClrBits );

	//
	// If the bitmap isn't compressed, set the BI_RGB flag
	//
	pbmi->bmiHeader.biCompression	= BI_RGB;

	//
	// Compute the number of bytes in the array of color indices
	// and store the result in biSizeImage.
	//
	pbmi->bmiHeader.biSizeImage		= ( pbmi->bmiHeader.biWidth + 7 ) / 8 * pbmi->bmiHeader.biHeight * cClrBits;

	//
	// Set biClrImportant to 0 to indicate that all of the device
	// colors are important.
	//
	pbmi->bmiHeader.biClrImportant	= 0;

	//
	// And finally return the info-structure.
	//
	return pbmi;
}

void ScreenShot(HWND hwnd)
{
	HDC hdcScreen;
	HDC hdcCompatible;
	HBITMAP	 hbmScreen;
	int zoom;
	char fname[256];
	char *WorkDir;

	zoom=getZoom();

    if(ProgDir) {
        if(WorkDir=getcwd(NULL,1024)) {
            chdir(ProgDir);
        }
    }

	hdcScreen=GetDC(hwnd);
	if (hdcScreen==NULL)
	{
	   MessageBox(NULL,"No hDC found!","",MB_OK);
	   return;
	}
	hdcCompatible = CreateCompatibleDC(hdcScreen);
	if (hdcCompatible==NULL)
	{
	   MessageBox(NULL,"Error creating Compatible DC","",MB_OK);
	   return;
	}
	
	hbmScreen     = CreateCompatibleBitmap(hdcScreen, WIDTH*zoom, (HEIGHT-1)*zoom);
	if (hbmScreen==NULL)
	{
	   MessageBox(NULL,"Error creating Compatible Bitmap","",MB_OK);
	   return;
	}

	SelectObject(hdcCompatible, hbmScreen);
	BitBlt(hdcCompatible, 0, 0, WIDTH*zoom, (HEIGHT-1)*zoom, hdcScreen, 0, 0, SRCCOPY);

	ttshots++; if (ttshots>9999) ttshots=1;
	sprintf(fname,"shots\\shot%04i.bmp",ttshots);
	if (SaveBitmap(fname, CreateBitmapInfoStructure(hbmScreen), hbmScreen, hdcCompatible)==E_FAIL)
	{
		MessageBox(NULL,"Error saving bitmap!","",MB_OK);
	}
	
	ReleaseDC(hwnd,hdcScreen);
	if (WorkDir) chdir(WorkDir);
}


#include "Common.h"
