#include "arch_Win32Properties.h"
#include <windows.h>
#include <commctrl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "resource.h"

static int propModified = 0;
static int canUseRegistry = 0;
static Mixer* theMixer;
extern void restartSound(int stereo);

static char virtualKeys[256][32] = {
    "",
    "", //"LBUTTON", 
    "", //"RBUTTON",
    "CANCEL",
    "", //"MBUTTON",
    "", //"XBUTTON1",
    "", //"XBUTTON2",
    "",
    "BACKSPACE",
    "", //"TAB",
    "",
    "",
    "CLEAR",
    "ENTER",
    "",
    "",
    
    "SHIFT",
    "CTRL",
    "ALT",
    "PAUSE",
    "CAPS LOCK",
    "KANA",
    "",
    "JUNJA",
    "FINAL",
    "KANJI",
    "",
    "ESC",
    "CONVERT",
    "NONCONVERT",
    "ACCEPT",
    "MODECHANGE",

    "SPACE",
    "PAGE UP",
    "PAGE DOWN",
    "END",
    "HOME",
    "LEFT ARROW",
    "UP ARROW",
    "RIGHT ARROW",
    "DOWN ARROW",
    "SELECT",
    "PRINT",
    "EXECUTE",
    "PRINT SCREEN",
    "INS",
    "DEL",
    "HELP",
    "0",
    "1",
    "2",
    "3",
    "4",
    "5",
    "6",
    "7",
    "8",
    "9",
    "",
    "",
    "",
    "",
    "",
    "",
    
    "",
    "A",
    "B",
    "C",
    "D",
    "E",
    "F",
    "G",
    "H",
    "I",
    "J",
    "K",
    "L",
    "M",
    "N",
    "O",

    "P",
    "Q",
    "R",
    "S",
    "T",
    "U",
    "V",
    "W",
    "X",
    "Y",
    "Z",
    "LEFT WIN",
    "RIGHT WIN",
    "APPS",
    "",
    "SLEEP",

    "NUMPAD 0",
    "NUMPAD 1",
    "NUMPAD 2",
    "NUMPAD 3",
    "NUMPAD 4",
    "NUMPAD 5",
    "NUMPAD 6",
    "NUMPAD 7",
    "NUMPAD 8",
    "NUMPAD 9",
    "NUMPAD *",
    "NUMPAD +",
    "NUMPAD ,",
    "NUMPAD -",
    "NUMPAD .",
    "NUMPAD /",

    "F1",
    "F2",
    "F3",
    "F4",
    "F5",
    "F6",
    "F7",
    "F8",
    "F9",
    "F10",
    "F11",
    "F12",
    "F13",
    "F14",
    "F15",
    "F16",

    "F17",
    "F18",
    "F19",
    "F20",
    "F21",
    "F22",
    "F23",
    "F24",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",

    "NUM LOCK",
    "SCROLL LOCK",
    "OEM 1",
    "OEM 2",
    "OEM 3",
    "OEM 4",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",

    "LEFT SHIFT",
    "RIGHT SHIFT",
    "LEFT CONTROL",
    "RIGHT CONTROL",
    "LEFT ALT",
    "RIGHT ALT",
    "BROWSER BACK",
    "BROWSER FORWARD",
    "BROWSER REFRESH",
    "BROWSER STOP",
    "BROWSER SEARCH",
    "BROWSER FAVORITES",
    "BROWSER HOME",
    "VOLUME MUTE",
    "VOLUME DOWN",
    "VOLUME UP",

    "NEXT TRACK",
    "PREV TRACK",
    "MEDIA STOP",
    "MEDIA PLAY",
    "LAUNCH MAIL",
    "LAUNCH MEDIA SELECT",
    "LAUNCH APP 1",
    "LAUNCH APP 2",
    "",
    "",
    ";",
    "+",
    ",",
    "-",
    ".",
    "?",

    "~",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",

    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "[",
    "\\",
    "]",
    "\"",
    "OEM 5",

    "",
    "OEM 6",
    "OEM 7",
    "OEM 8",
    "OEM 9",
    "PROCESS",
    "OEM 10",
    "",
    "",
    "OEM 11",
    "OEM 12",
    "OEM 13",
    "OEM 14",
    "OEM 15",
    "OEM 16",
    "OEM 17",

    "OEM 18",
    "OEM 19",
    "OEM 20",
    "OEM 21",
    "OEM 22",
    "OEM 23",
    "ATTN",
    "CRSEL",
    "EXSEL",
    "EREOF",
    "PLAY",
    "ZOOM",
    "",
    "PA1",
    "CLEAR",
    ""
};

static char registryKey[] = "RedMSX";

static char* pEmuFamily[] = {
    "MSX 1", 
    "MSX 2", 
    "MSX 2+", 
    "MSX 1 - Brazilian",
    "MSX 1 - Japanese", 
    "MSX 2 - Japanese", 
    NULL 
};

static char* pEmuSpeed[] = {
    "1.193 kHz (a third)",
    "1.790 MHz (half)",
    "3.579 MHz (normal)",
    "7.158 MHz (double)",
    "10.737 MHz (tripple)",
    NULL
};

static char* pEmuRAM[] = {
    "16 kBytes",
    "32 kBytes",
    "64 kBytes",
    "128 kBytes",
    "256 kBytes",
    "512 kBytes",
    "1 MBytes",
    "2 MBytes",
    "4 MBytes",
    NULL
};

static char* pEmuVRAM[] = {
    "16 kBytes",
    "32 kBytes",
    "64 kBytes",
    "128 kBytes",
	"256 kBytes",
    NULL
};

static char* pEmuSync[] = {
    "1 ms (best)",
    "Auto (fast)",
    NULL
};

static char* pVideoMon[] = {
    "Color",
    "Black and white",
    "Green",
    NULL
};

static char* pVideoVideoType[] = {
    "PAL",
    "NTSC",
    NULL
};

static char* pVideoPalEmu[] = {
    "None",
    "Y/C cable (sharp)",
    "Noisy Y/C cable (sharp)",
    "Composite (blurry)",
    "Noisy Composite (blurry)",
	"Scale 2x",
    NULL
};

static char* pVideoMonSize[] = {
    "Normal - 320x200",
    "Double - 640x400",
    "Fullscreen",
    NULL
};

static char* pVideoDriver[] = {
    "DirectDraw",
    "GDI",
    NULL
};

static char* pVideoFrameSkip[] = {
    "None",
    "1 Frame",
    "2 Frames",
    "3 Frames",
    "4 Frames",
    "5 Frames",
    NULL
};

static char* pVideoFullRes[] = {
    "320x240 16bit",
    "320x240 32bit",
    "640x480 16bit",
    "640x480 32bit",
    NULL
};

static char* pSoundDriver[] = {
    "No Sound",
    "WMM driver",
    "Direct X driver",
    NULL
};

static char* pSoundFrequency[] = {
    "11025 Hz",
    "22050 Hz",
    "44100 Hz",
    NULL
};

static char* pSoundBufferSize[] = {
    "100 ms",
    "150 ms",
    "200 ms",
    "250 ms",
    "300 ms",
    "350 ms",
    NULL
};

static char* pControlsJoy[] = {
    "None",
    "Numpad",
    "Keyset A",
    "Keyset B",
    "PC joystick #1",
    "PC joystick #2",
    "Mouse",
    NULL
};

static char* pControlsAutofire[] = {
    "Off",
    "Slow",
    "Medium",
    "Fast",
    NULL
};

static char* pControlsBiosCharSet[] = {
    "European",
//    "Russian",
//    "Japanese",
//    "Korean",
    NULL
};


static void getRegStrValue(char* keyDir, char* keyStr, char* returnValue) {  
    char value[1024];
    LONG rv;
    HKEY hKey;
    DWORD length = 1024;
    DWORD type;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegOpenKeyEx(HKEY_CURRENT_USER, directory, 0, KEY_QUERY_VALUE, &hKey);    
    if (rv != ERROR_SUCCESS) {
        return;
    }

    rv = RegQueryValueEx(hKey, keyStr, NULL, &type, NULL, &length);

    rv = RegQueryValueEx(hKey, keyStr, NULL, &type, (BYTE*)value, &length);
    RegCloseKey(hKey);
    if (rv != ERROR_SUCCESS) {
        return;
    }

    strcpy(returnValue, value);
}

static void getRegIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {  
    LONG rv;
    HKEY hKey;
    DWORD length;
    DWORD type;
    DWORD value;
    char directory[32];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegOpenKeyEx(HKEY_CURRENT_USER, directory, 0, KEY_QUERY_VALUE, &hKey);  
    if (rv != ERROR_SUCCESS) {
        return;
    }

    rv = RegQueryValueEx(hKey, keyStr, NULL, &type, NULL, &length);

	rv = RegQueryValueEx(hKey, keyStr, NULL, &type, (BYTE *)&value, &length);

    RegCloseKey(hKey);

    if (rv != ERROR_SUCCESS) {
        return;
    }

    *returnValue = value;
}

static BOOL setRegIntValue(char* keyDir, char* keyStr, DWORD value) {
    HKEY hKey;
    DWORD exist;
    DWORD rv;
    DWORD dwValue = (DWORD)value;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegCreateKeyEx(HKEY_CURRENT_USER, directory, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    if (rv != ERROR_SUCCESS) {
        return FALSE;
    }

    rv = RegSetValueEx(hKey, keyStr, 0, REG_DWORD, (BYTE *)&dwValue, sizeof(DWORD));

    RegCloseKey(hKey);

    return TRUE;
}

static BOOL setRegStrValue(char* keyDir, char* keyStr, char* value) {
    HKEY hKey;
    DWORD exist;
    DWORD rv;
    DWORD dwValue = (DWORD)value;
    char directory[64];

    sprintf(directory, "Software\\%s", keyDir);

    rv = RegCreateKeyEx(HKEY_CURRENT_USER, directory, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &exist);
    if (rv != ERROR_SUCCESS) {
        return FALSE;
    }

    rv = RegSetValueEx(hKey, keyStr, 0, REG_SZ, (BYTE *)value, strlen(value) + 1);

    RegCloseKey(hKey);

    return TRUE;
}

static void getIniStrValue(char* keyDir, char* keyStr, char* returnValue) {  
    char file[128];
    char defStr[128];

    sprintf(file, "%s.ini", keyDir);
    strcpy(defStr, returnValue);

    GetPrivateProfileString("General", keyStr, defStr, returnValue, 64, file);      
}

static void getIniIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {  
    char file[128];
    DWORD def = *returnValue;

    sprintf(file, "%s.ini", keyDir);

    *returnValue = GetPrivateProfileInt("General", keyStr, def, file);                           
}

static BOOL setIniIntValue(char* keyDir, char* keyStr, DWORD value) {
    char file[128];
    char buf[30];

    sprintf(buf, "%d", value);
    sprintf(file, "%s.ini", keyDir);

    return WritePrivateProfileString("General", keyStr, buf, file);
}

static BOOL setIniStrValue(char* keyDir, char* keyStr, char* value) {
    char file[128];
    sprintf(file, "%s.ini", keyDir);
    
    return WritePrivateProfileString("General", keyStr, value, file);
}
  
static void getStrValue(char* keyDir, char* keyStr, char* returnValue) {
    if (canUseRegistry>0) getRegStrValue(keyDir, keyStr, returnValue);
    else                getIniStrValue(keyDir, keyStr, returnValue);
}

static void getIntValue(char* keyDir, char* keyStr, DWORD* returnValue) {
    if (canUseRegistry>0) getRegIntValue(keyDir, keyStr, returnValue);
    else                getIniIntValue(keyDir, keyStr, returnValue);
}   
    
static BOOL setStrValue(char* keyDir, char* keyStr, char* value) {
    if (canUseRegistry>0) return setRegStrValue(keyDir, keyStr, value);
    else                return setIniStrValue(keyDir, keyStr, value);
}

static BOOL setIntValue(char* keyDir, char* keyStr, DWORD value) {
    if (canUseRegistry>0) return setRegIntValue(keyDir, keyStr, value);
    else                return setIniIntValue(keyDir, keyStr, value);
}

static void initDropList(HWND hDlg, int id, char** pList, int index) {
    while (*pList != NULL) {
        SendDlgItemMessage(hDlg, id, CB_ADDSTRING, 0, (LPARAM)*pList);
        pList++;
    }

    SendDlgItemMessage(hDlg, id, CB_SETCURSEL, index, 0);
}

static int getDropListIndex(HWND hDlg, int id, char** pList) {
    int index = 0;
    char s[64];

    GetDlgItemText(hDlg, id, s, 63);
    
    while (*pList != NULL) {
        if (0 == strcmp(s, *pList)) {
            return index;
        }
        index++;
        pList++;
    }

    return -1;
}

static BOOL CALLBACK emulationDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        initDropList(hDlg, IDC_EMUFAMILY, pEmuFamily, pProperties->emulation.family);
        initDropList(hDlg, IDC_EMUSPEED, pEmuSpeed, pProperties->emulation.speed);
        initDropList(hDlg, IDC_EMURAM, pEmuRAM, pProperties->emulation.RAMsize);
        initDropList(hDlg, IDC_EMUVRAM, pEmuVRAM, pProperties->emulation.VRAMsize);
        initDropList(hDlg, IDC_EMUSYNC, pEmuSync, pProperties->emulation.syncMethod);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }
            
        pProperties->emulation.family       = getDropListIndex(hDlg, IDC_EMUFAMILY, pEmuFamily);
        pProperties->emulation.speed        = getDropListIndex(hDlg, IDC_EMUSPEED, pEmuSpeed);
        pProperties->emulation.RAMsize      = getDropListIndex(hDlg, IDC_EMURAM, pEmuRAM);
        pProperties->emulation.VRAMsize     = getDropListIndex(hDlg, IDC_EMUVRAM, pEmuVRAM);
        pProperties->emulation.syncMethod   = getDropListIndex(hDlg, IDC_EMUSYNC, pEmuSync);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

static BOOL CALLBACK videoDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        initDropList(hDlg, IDC_MONTYPE, pVideoMon, pProperties->video.monType);
        initDropList(hDlg, IDC_PALEMU, pVideoPalEmu, pProperties->video.palEmu);
        initDropList(hDlg, IDC_VIDEOTYPE, pVideoVideoType, pProperties->video.videoType);
        initDropList(hDlg, IDC_MONSIZE, pVideoMonSize, pProperties->video.size);
        initDropList(hDlg, IDC_VIDEODRV, pVideoDriver, pProperties->video.driver);
        initDropList(hDlg, IDC_FRAMESKIP, pVideoFrameSkip, pProperties->video.frameSkip);
//        initDropList(hDlg, IDC_FULLRES, pVideoFullRes, pProperties->video.fullRes);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }

        pProperties->video.monType          = getDropListIndex(hDlg, IDC_MONTYPE, pVideoMon);
        pProperties->video.palEmu           = getDropListIndex(hDlg, IDC_PALEMU, pVideoPalEmu);
        pProperties->video.videoType        = getDropListIndex(hDlg, IDC_VIDEOTYPE, pVideoVideoType);
        pProperties->video.size             = getDropListIndex(hDlg, IDC_MONSIZE, pVideoMonSize);
        pProperties->video.driver           = getDropListIndex(hDlg, IDC_VIDEODRV, pVideoDriver);
        pProperties->video.frameSkip        = getDropListIndex(hDlg, IDC_FRAMESKIP, pVideoFrameSkip);
//        pProperties->video.fullRes          = getDropListIndex(hDlg, IDC_FULLRES, pVideoFullRes);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

static void CreateToolTip(HWND hwndCtrl, HWND* hwndTT, TOOLINFO* ti)
{
    *hwndTT = CreateWindow(TOOLTIPS_CLASS, TEXT(""), WS_POPUP, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
                          NULL, (HMENU)NULL, GetModuleHandle(NULL), NULL);

    ti->cbSize = sizeof(TOOLINFO);
    ti->uFlags = TTF_IDISHWND | TTF_CENTERTIP | TTF_ABSOLUTE;
    ti->hwnd   = hwndCtrl;
    ti->uId    = (UINT)hwndCtrl;
    ti->hinst  = GetModuleHandle(NULL);
    ti->lpszText  = "";
    ti->rect.left = ti->rect.top = ti->rect.bottom = ti->rect.right = 60; 

    SendMessage(*hwndTT, TTM_ADDTOOL, 0, (LPARAM)ti);
    SendMessage(*hwndTT, TTM_TRACKACTIVATE, (WPARAM)TRUE, (LPARAM)ti);
    SendMessage(hwndCtrl, TBM_SETTOOLTIPS, (WPARAM)*hwndTT, 0);
}

static void UpdateVolumeToolTip(HWND hCtrl, HWND hwndTT, TOOLINFO* ti)
{
    static char str[32];
    int val = SendMessage(hCtrl, TBM_GETPOS,   0, 0);
    if (val == 100) {
        sprintf(str, "-inf.");
    }
    else {
        sprintf(str, "%.1f dB", -30. * val / 100);
    }
    ti->lpszText  = str;
    SendMessage(hwndTT, TTM_UPDATETIPTEXT, 0, (LPARAM)ti);  
}

static void UpdatePanToolTip(HWND hCtrl, HWND hwndTT, TOOLINFO* ti)
{
    static char str[32];
    int val = SendMessage(hCtrl, TBM_GETPOS,   0, 0);
    if (val == 50) {
        sprintf(str, "    C    ");
    }
    else if (val < 50) {
        sprintf(str, "L: %d", 2 * (50 - val));
    }
    else {
        sprintf(str, "R: %d", 2 * (val - 50));
    }
    ti->lpszText  = str;
    SendMessage(hwndTT, TTM_UPDATETIPTEXT, 0, (LPARAM)ti);  
}

static BOOL CALLBACK soundDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static HWND hwndVolTT[MCT_NUMENTRIES];
    static TOOLINFO tiVol[MCT_NUMENTRIES];
    static HWND hwndPanTT[MCT_NUMENTRIES];
    static TOOLINFO tiPan[MCT_NUMENTRIES];
    static HWND hwndMasterTT[2];
    static TOOLINFO tiMaster[2];
    static Properties* pProperties;
    static int stereo;
    int updateAudio = FALSE;
    int i;

    switch (iMsg) {
    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        initDropList(hDlg, IDC_SNDDRIVER, pSoundDriver, pProperties->sound.driver);
        initDropList(hDlg, IDC_SNDFREQ, pSoundFrequency, pProperties->sound.frequency);
        initDropList(hDlg, IDC_SNDBUFSZ, pSoundBufferSize, pProperties->sound.bufSize);

        for (i = 0; i < MCT_NUMENTRIES; i++) {
            CreateToolTip(GetDlgItem(hDlg, IDC_VOLUME1 + i), &hwndVolTT[i], &tiVol[i]);
            CreateToolTip(GetDlgItem(hDlg, IDC_PAN1 + i), &hwndPanTT[i], &tiPan[i]);

            SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
            SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.mixerChannel[i].volume));
            SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
            SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_SETPOS,   1, (LPARAM)pProperties->sound.mixerChannel[i].pan);
            SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_SETCHECK, pProperties->sound.mixerChannel[i].enable ? BST_CHECKED : BST_UNCHECKED, 0);
        }

        CreateToolTip(GetDlgItem(hDlg, IDC_MASTERL), &hwndMasterTT[0], &tiMaster[0]);
        CreateToolTip(GetDlgItem(hDlg, IDC_MASTERR), &hwndMasterTT[1], &tiMaster[1]);

        SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.masterVolume));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETRANGE, 0, (LPARAM)MAKELONG(0, 100));
        SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)(100 - pProperties->sound.masterVolume));

        stereo = pProperties->sound.stereo;
        SetWindowText(GetDlgItem(hDlg, IDC_STEREO), stereo ? "stereo" : "mono");

        return FALSE;

    case WM_COMMAND:
        if (LOWORD(wParam) >= IDC_VOLENABLE1 && LOWORD(wParam) < IDC_VOLENABLE1 + MCT_NUMENTRIES) {
            updateAudio = TRUE;
        }

        switch(LOWORD(wParam)) {
        case IDC_STEREO:
            stereo = !stereo;
            SetWindowText(GetDlgItem(hDlg, IDC_STEREO), stereo ? "stereo" : "mono");
            restartSound(stereo);
            updateAudio = TRUE;
            break;
        case IDC_MASTERL:
            SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
            updateAudio = TRUE;
            break;
        case IDC_MASTERR:
            SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_GETPOS,   0, 0));
            updateAudio = TRUE;
            break;
        }
        break;

    case WM_NOTIFY:
        switch (((NMHDR FAR *)lParam)->code) {
        case PSN_APPLY:
            pProperties->sound.driver           = getDropListIndex(hDlg, IDC_SNDDRIVER, pSoundDriver);
            pProperties->sound.frequency        = getDropListIndex(hDlg, IDC_SNDFREQ, pSoundFrequency);
            pProperties->sound.bufSize          = getDropListIndex(hDlg, IDC_SNDBUFSZ, pSoundBufferSize);

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                pProperties->sound.mixerChannel[i].volume = 100 - SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i),    TBM_GETPOS,   0, 0);
                pProperties->sound.mixerChannel[i].pan    = SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i),       TBM_GETPOS,   0, 0);
                pProperties->sound.mixerChannel[i].enable = SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_GETCHECK, 0, 0) == BST_CHECKED;
            }

            pProperties->sound.masterVolume = 100 - SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0);
            pProperties->sound.stereo = stereo;

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                DestroyWindow(hwndVolTT[i]);
                DestroyWindow(hwndPanTT[i]);
            }
            DestroyWindow(hwndMasterTT[0]);
            DestroyWindow(hwndMasterTT[1]);

            propModified = 1;
            return TRUE;

        case PSN_QUERYCANCEL:
            restartSound(pProperties->sound.stereo);

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                mixerSetChannelVolume(theMixer, i, pProperties->sound.mixerChannel[i].volume);
                mixerSetChannelPan(theMixer, i, pProperties->sound.mixerChannel[i].pan);
                mixerEnableChannel(theMixer, i, pProperties->sound.mixerChannel[i].enable);
            }
            mixerSetMasterVolume(theMixer, pProperties->sound.masterVolume);

            for (i = 0; i < MCT_NUMENTRIES; i++) {
                DestroyWindow(hwndVolTT[i]);
                DestroyWindow(hwndPanTT[i]);
            }
            DestroyWindow(hwndMasterTT[0]);
            DestroyWindow(hwndMasterTT[1]);

            return FALSE;

        default:
            if (wParam == IDC_MASTERL) {
                SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
                UpdateVolumeToolTip(GetDlgItem(hDlg, IDC_MASTERL), hwndMasterTT[0], &tiMaster[0]);
                updateAudio = TRUE;
            }
            if (wParam == IDC_MASTERR) {
                SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_SETPOS,   1, (LPARAM)SendMessage(GetDlgItem(hDlg, IDC_MASTERR), TBM_GETPOS,   0, 0));
                UpdateVolumeToolTip(GetDlgItem(hDlg, IDC_MASTERR), hwndMasterTT[1], &tiMaster[1]);
                updateAudio = TRUE;
            }

            if (wParam >= IDC_VOLUME1 && wParam < IDC_VOLUME1 + MCT_NUMENTRIES) {  
                UpdateVolumeToolTip(GetDlgItem(hDlg, wParam), hwndVolTT[wParam - IDC_VOLUME1], &tiVol[wParam - IDC_VOLUME1]);
                updateAudio = TRUE;
            }

            if (wParam >= IDC_PAN1 && wParam < IDC_PAN1 + MCT_NUMENTRIES) {    
                UpdatePanToolTip(GetDlgItem(hDlg, wParam), hwndPanTT[wParam - IDC_PAN1], &tiPan[wParam - IDC_PAN1]);
                updateAudio = TRUE;
            }

            break;
        }

        break;
    }

    if (updateAudio) {
        for (i = 0; i < MCT_NUMENTRIES; i++) {
            mixerSetChannelVolume(theMixer, i, 100 - SendMessage(GetDlgItem(hDlg, IDC_VOLUME1 + i), TBM_GETPOS,   0, 0));
            mixerSetChannelPan(theMixer, i, SendMessage(GetDlgItem(hDlg, IDC_PAN1 + i), TBM_GETPOS,   0, 0));
            mixerEnableChannel(theMixer, i, SendMessage(GetDlgItem(hDlg, IDC_VOLENABLE1 + i), BM_GETCHECK, 0, 0) == BST_CHECKED);
        }
        mixerSetMasterVolume(theMixer, 100 - SendMessage(GetDlgItem(hDlg, IDC_MASTERL), TBM_GETPOS,   0, 0));
    }

    return FALSE;
}

static BOOL CALLBACK joykeyDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static JoystickProperties* joyCtrl;
    static JoystickProperties tmpJoyCtrl;
    static int id = 0;

    switch (iMsg) {
    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDOK:
            KillTimer(hDlg, 0);
            *joyCtrl = tmpJoyCtrl;
            EndDialog(hDlg, TRUE);
            return TRUE;
        case IDCANCEL:
            KillTimer(hDlg, 0);
            EndDialog(hDlg, FALSE);
            return TRUE;
        case IDC_KEYUP:
            break;
        }
        switch (HIWORD(wParam)) {
        case EN_KILLFOCUS:
            id = 0;
            break;
        case EN_SETFOCUS:
            id = LOWORD(wParam);
            break;
        }
        break;
    case WM_TIMER:
        if (id != 0) {
            int i;
            for (i = 0; i < 256; i++) {
                if (virtualKeys[i][0]) {
                    SHORT state = GetAsyncKeyState(i);
                    if (state & 1) {
                        SendMessage(GetDlgItem(hDlg, id), WM_SETTEXT, 0, (LPARAM)virtualKeys[i]);
                        switch (id) {
                        case IDC_KEYUP:      tmpJoyCtrl.keyUp    = i; break;
                        case IDC_KEYDOWN:    tmpJoyCtrl.keyDown  = i; break;
                        case IDC_KEYLEFT:    tmpJoyCtrl.keyLeft  = i; break;
                        case IDC_KEYRIGHT:   tmpJoyCtrl.keyRight = i; break;
                        case IDC_KEYBUTTON1: tmpJoyCtrl.button1  = i; break;
                        case IDC_KEYBUTTON2: tmpJoyCtrl.button2  = i; break;
                        }
                    }
                }
            }
        }
        break;

    case WM_INITDIALOG:
        joyCtrl = (JoystickProperties*)lParam;
        {
            char title[64];
            sprintf(title, "blueMSX - Joystick keyset %c", 'A' + joyCtrl->id - 1);
            SetWindowText(hDlg, title);
        }
        tmpJoyCtrl = *joyCtrl;
        SendMessage(GetDlgItem(hDlg, IDC_KEYUP),      WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyUp]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYDOWN),    WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyDown]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYLEFT),    WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyLeft]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYRIGHT),   WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.keyRight]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYBUTTON1), WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.button1]);
        SendMessage(GetDlgItem(hDlg, IDC_KEYBUTTON2), WM_SETTEXT, 0, (LPARAM)virtualKeys[tmpJoyCtrl.button2]);

        SetTimer(hDlg, 0, 1, NULL);
        return FALSE;
    }

    return FALSE;
}

static BOOL CALLBACK controlsDlgProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam) {
    static Properties* pProperties;

    switch (iMsg) {
    case WM_COMMAND:
        switch(LOWORD(wParam)) {
        case IDC_KEYSET1:
            DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_JOYKEYS), hDlg, joykeyDlgProc, (LPARAM)&pProperties->joy1);
            return TRUE;
        case IDC_KEYSET2:
            DialogBoxParam(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_JOYKEYS), hDlg, joykeyDlgProc, (LPARAM)&pProperties->joy2);
            return TRUE;
        }
        break;

    case WM_INITDIALOG:
        pProperties = (Properties*)((PROPSHEETPAGE*)lParam)->lParam;

        initDropList(hDlg, IDC_JOY1, pControlsJoy, pProperties->joy1.type);
        initDropList(hDlg, IDC_AUTOFIRE1, pControlsAutofire, pProperties->joy1.autofire);
        initDropList(hDlg, IDC_JOY2, pControlsJoy, pProperties->joy2.type);
        initDropList(hDlg, IDC_AUTOFIRE2, pControlsAutofire, pProperties->joy2.autofire);
        initDropList(hDlg, IDC_CHARSET, pControlsBiosCharSet, pProperties->keyboard.keySet);

        return FALSE;

    case WM_NOTIFY:
        if ((((NMHDR FAR *)lParam)->code) != PSN_APPLY) {
            return FALSE;
        }

        pProperties->joy1.type       = getDropListIndex(hDlg, IDC_JOY1, pControlsJoy);
        pProperties->joy1.autofire   = getDropListIndex(hDlg, IDC_AUTOFIRE1, pControlsAutofire);
        pProperties->joy2.type       = getDropListIndex(hDlg, IDC_JOY2, pControlsJoy);
        pProperties->joy2.autofire   = getDropListIndex(hDlg, IDC_AUTOFIRE2, pControlsAutofire);
        pProperties->keyboard.keySet = getDropListIndex(hDlg, IDC_CHARSET, pControlsBiosCharSet);

        propModified = 1;
        
        return TRUE;
    }

    return FALSE;
}

int showProperties(Properties* pProperties, HWND hwndOwner, PropPage startPage, Mixer* mixer) {
	HINSTANCE       hInst = (HINSTANCE)GetModuleHandle(NULL);
    PROPSHEETPAGE   psp[4];
    PROPSHEETHEADER psh;
    Properties oldProp = *pProperties;

    theMixer = mixer;

    psp[0].dwSize = sizeof(PROPSHEETPAGE);
    psp[0].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[0].hInstance = hInst;
    psp[0].pszTemplate = MAKEINTRESOURCE(IDD_EMULATION);
    psp[0].pszIcon = NULL;
    psp[0].pfnDlgProc = emulationDlgProc;
    psp[0].pszTitle = "Emulation";
    psp[0].lParam = (LPARAM)pProperties;
    psp[0].pfnCallback = NULL;

    psp[1].dwSize = sizeof(PROPSHEETPAGE);
    psp[1].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[1].hInstance = hInst;
    psp[1].pszTemplate = MAKEINTRESOURCE(IDD_VIDEO);
    psp[1].pszIcon = NULL;
    psp[1].pfnDlgProc = videoDlgProc;
    psp[1].pszTitle = "Video";
    psp[1].lParam = (LPARAM)pProperties;
    psp[1].pfnCallback = NULL;

    psp[2].dwSize = sizeof(PROPSHEETPAGE);
    psp[2].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[2].hInstance = hInst;
    psp[2].pszTemplate = MAKEINTRESOURCE(IDD_SOUND);
    psp[2].pszIcon = NULL;
    psp[2].pfnDlgProc = soundDlgProc;
    psp[2].pszTitle = "Sound";
    psp[2].lParam = (LPARAM)pProperties;
    psp[2].pfnCallback = NULL;

    psp[3].dwSize = sizeof(PROPSHEETPAGE);
    psp[3].dwFlags = PSP_USEICONID | PSP_USETITLE;
    psp[3].hInstance = hInst;
    psp[3].pszTemplate = MAKEINTRESOURCE(IDD_CONTROLS);
    psp[3].pszIcon = NULL;
    psp[3].pfnDlgProc = controlsDlgProc;
    psp[3].pszTitle = "Controls";
    psp[3].lParam = (LPARAM)pProperties;
    psp[3].pfnCallback = NULL;

    psh.dwSize = sizeof(PROPSHEETHEADER);
    psh.dwFlags = PSH_USEICONID | PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
    psh.hwndParent = hwndOwner;
    psh.hInstance = hInst;
    psh.pszIcon = NULL;
    psh.pszCaption = (LPSTR) "blueMSX - Properties";
    psh.nPages = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.nStartPage = startPage;
    psh.ppsp = (LPCPROPSHEETPAGE) &psp;
    psh.pfnCallback = NULL;

    propModified = 0;

    PropertySheet(&psh);

    if (propModified) {
        propModified = memcmp(&oldProp, pProperties, sizeof(Properties));
    }

    return propModified;
}

void propInitDefaults(Properties* pProperties) 
{
    int i;
    pProperties->emulation.family       = P_EMU_MSX2;
    pProperties->emulation.speed        = P_EMU_SPEEDX1;
    pProperties->emulation.RAMsize      = P_EMU_RAM128;
    pProperties->emulation.VRAMsize     = P_EMU_VRAM128;
    pProperties->emulation.syncMethod   = P_EMU_SYNCAUTO;
    
    pProperties->video.monType          = P_VIDEO_COLOR;
    pProperties->video.palEmu           = P_VIDEO_PALSCALE2X;
    pProperties->video.videoType        = P_VIDEO_PAL;
    pProperties->video.size             = P_VIDEO_SIZEX2;
    pProperties->video.driver           = P_VIDEO_DRVDIRECTX;
    pProperties->video.frameSkip        = P_VIDEO_FSKIP0;
    pProperties->video.fullRes          = P_VIDEO_FRES640X480_32;

    pProperties->sound.driver           = P_SOUND_DRVDIRECTX;
    pProperties->sound.frequency        = P_SOUND_FREQ44;
    pProperties->sound.bufSize          = P_SOUND_BUF200;
    pProperties->sound.syncMethod       = P_SOUND_SYNCQADJUST;
    pProperties->sound.logFileCount     = 0;

    pProperties->sound.stereo = 1;
    pProperties->sound.masterVolume = 82;

    pProperties->sound.mixerChannel[MCT_PSG].enable = 1;
    pProperties->sound.mixerChannel[MCT_PSG].pan = 42;
    pProperties->sound.mixerChannel[MCT_PSG].volume = 100;

    pProperties->sound.mixerChannel[MCT_SCC].enable = 1;
    pProperties->sound.mixerChannel[MCT_SCC].pan = 58;
    pProperties->sound.mixerChannel[MCT_SCC].volume = 86;

    pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable = 1;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan = 58;
    pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume = 100;

    pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable = 0;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan = 50;
    pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume = 0;

    pProperties->sound.mixerChannel[MCT_KEYBOARD].enable = 1;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].pan = 54;
    pProperties->sound.mixerChannel[MCT_KEYBOARD].volume = 45;

    pProperties->sound.mixerChannel[MCT_CASETTE].enable = 0;
    pProperties->sound.mixerChannel[MCT_CASETTE].pan = 50;
    pProperties->sound.mixerChannel[MCT_CASETTE].volume = 0;
    
    pProperties->joy1.type              = P_JOY_NUMPAD;
    pProperties->joy1.autofire          = P_JOY_AFOFF;
    pProperties->joy1.keyUp             = 0xff;
    pProperties->joy1.keyDown           = 0xff;
    pProperties->joy1.keyLeft           = 0xff;
    pProperties->joy1.keyRight          = 0xff;
    pProperties->joy1.button1           = 0xff;
    pProperties->joy1.button2           = 0xff;
    
    pProperties->joy2.type              = P_JOY_NONE;
    pProperties->joy2.autofire          = P_JOY_AFOFF;
    pProperties->joy2.keyUp             = 0xff;
    pProperties->joy2.keyDown           = 0xff;
    pProperties->joy2.keyLeft           = 0xff;
    pProperties->joy2.keyRight          = 0xff;
    pProperties->joy2.button1           = 0xff;
    pProperties->joy2.button2           = 0xff;
    
    pProperties->keyboard.keySet        = P_CHAR_EUROPEAN;
    
    pProperties->cartridge.slotA[0]     = 0;
    pProperties->cartridge.slotB[0]     = 0;
    //pProperties->cartridge.slotAZip[0]  = 0;
    //pProperties->cartridge.slotBZip[0]  = 0;
    pProperties->diskdrive.slotA[0]     = 0;
    pProperties->diskdrive.slotB[0]     = 0;
    //pProperties->diskdrive.slotAZip[0]  = 0;
    //pProperties->diskdrive.slotBZip[0]  = 0;
    pProperties->diskdrive.autostartA   = 0;

    for (i = 0; i < MAX_HISTORY; i++) {
        pProperties->filehistory.cartridgeA[i][0] = 0;
        pProperties->filehistory.cartridgeB[i][0] = 0;
        pProperties->filehistory.diskdriveA[i][0] = 0;
        pProperties->filehistory.diskdriveB[i][0] = 0;
    }

}

void propLoad(Properties* pProperties) 
{
    int i;
    getIntValue(registryKey, "emulation-family", (DWORD*)&pProperties->emulation.family);
    getIntValue(registryKey, "emulation-speed", (DWORD*)&pProperties->emulation.speed);
    getIntValue(registryKey, "emulation-ramsize", (DWORD*)&pProperties->emulation.RAMsize);
    getIntValue(registryKey, "emulation-vramsize", (DWORD*)&pProperties->emulation.VRAMsize);
    getIntValue(registryKey, "emulation-syncmethod", (DWORD*)&pProperties->emulation.syncMethod);
    
    getIntValue(registryKey, "video-montype", (DWORD*)&pProperties->video.monType);
    getIntValue(registryKey, "video-palemu", (DWORD*)&pProperties->video.palEmu);
    getIntValue(registryKey, "video-type", (DWORD*)&pProperties->video.videoType);
    getIntValue(registryKey, "video-size", (DWORD*)&pProperties->video.size);
    getIntValue(registryKey, "video-driver", (DWORD*)&pProperties->video.driver);
    getIntValue(registryKey, "video-frameskip", (DWORD*)&pProperties->video.frameSkip);
    getIntValue(registryKey, "video-fullres", (DWORD*)&pProperties->video.fullRes);
    
    getIntValue(registryKey, "sound-driver", (DWORD*)&pProperties->sound.driver);
    getIntValue(registryKey, "sound-frequency", (DWORD*)&pProperties->sound.frequency);
    getIntValue(registryKey, "sound-bufsize", (DWORD*)&pProperties->sound.bufSize);
    getIntValue(registryKey, "sound-syncmethod", (DWORD*)&pProperties->sound.syncMethod);
    getIntValue(registryKey, "sound-logfilecount", (DWORD*)&pProperties->sound.logFileCount);

    getIntValue(registryKey, "sound-stereo", (DWORD*)&pProperties->sound.stereo);
    getIntValue(registryKey, "sound-mastervolume", (DWORD*)&pProperties->sound.masterVolume);
    getIntValue(registryKey, "sound-mcPSGenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].enable);
    getIntValue(registryKey, "sound-mcPSGpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].pan);
    getIntValue(registryKey, "sound-mcPSGvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_PSG].volume);
    getIntValue(registryKey, "sound-mcSCCenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].enable);
    getIntValue(registryKey, "sound-mcSCCpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].pan);
    getIntValue(registryKey, "sound-mcSCCvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_SCC].volume);
    getIntValue(registryKey, "sound-mcAUDIOenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable);
    getIntValue(registryKey, "sound-mcAUDIOpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan);
    getIntValue(registryKey, "sound-mcAUDIOvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume);
    getIntValue(registryKey, "sound-mcMUSICenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable);
    getIntValue(registryKey, "sound-mcMUSICpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan);
    getIntValue(registryKey, "sound-mcMUSICvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume);
    getIntValue(registryKey, "sound-mcKEYBOARDenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].enable);
    getIntValue(registryKey, "sound-mcKEYBOARDpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].pan);
    getIntValue(registryKey, "sound-mcKEYBOARDvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_KEYBOARD].volume);
    getIntValue(registryKey, "sound-mcCASETTEenable", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASETTE].enable);
    getIntValue(registryKey, "sound-mcCASETTEpan", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASETTE].pan);
    getIntValue(registryKey, "sound-mcCASETTEvolume", (DWORD*)&pProperties->sound.mixerChannel[MCT_CASETTE].volume);
   
    getIntValue(registryKey, "joy1-type", (DWORD*)&pProperties->joy1.type);
    getIntValue(registryKey, "joy1-autofire", (DWORD*)&pProperties->joy1.autofire);
    getIntValue(registryKey, "joy1-keyup", (DWORD*)&pProperties->joy1.keyUp);
    getIntValue(registryKey, "joy1-keydown", (DWORD*)&pProperties->joy1.keyDown);
    getIntValue(registryKey, "joy1-keyleft", (DWORD*)&pProperties->joy1.keyLeft);
    getIntValue(registryKey, "joy1-keyright", (DWORD*)&pProperties->joy1.keyRight);
    getIntValue(registryKey, "joy1-button1", (DWORD*)&pProperties->joy1.button1);
    getIntValue(registryKey, "joy1-button2", (DWORD*)&pProperties->joy1.button2);
    
    getIntValue(registryKey, "joy2-type", (DWORD*)&pProperties->joy2.type);
    getIntValue(registryKey, "joy2-autofire", (DWORD*)&pProperties->joy2.autofire);
    getIntValue(registryKey, "joy2-keyup", (DWORD*)&pProperties->joy2.keyUp);
    getIntValue(registryKey, "joy2-keydown", (DWORD*)&pProperties->joy2.keyDown);
    getIntValue(registryKey, "joy2-keyleft", (DWORD*)&pProperties->joy2.keyLeft);
    getIntValue(registryKey, "joy2-keyright", (DWORD*)&pProperties->joy2.keyRight);
    getIntValue(registryKey, "joy2-button1", (DWORD*)&pProperties->joy2.button1);
    getIntValue(registryKey, "joy2-button2", (DWORD*)&pProperties->joy2.button2);
    
    getIntValue(registryKey, "keyboard-keyset", (DWORD*)&pProperties->keyboard.keySet);

    getStrValue(registryKey, "cartridge-slota", (char*)pProperties->cartridge.slotA);
    getStrValue(registryKey, "cartridge-slotb", (char*)pProperties->cartridge.slotB);

    getStrValue(registryKey, "diskdrive-slota", (char*)pProperties->diskdrive.slotA);
    getStrValue(registryKey, "diskdrive-slotb", (char*)pProperties->diskdrive.slotB);
    getIntValue(registryKey, "diskdrive-autostarta", (DWORD*)&pProperties->diskdrive.autostartA);
    
    for (i = 0; i < MAX_HISTORY; i++) {
        char str[32];
        sprintf(str, "filehist-carta %X", i);
        getStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeA[i]);
        sprintf(str, "filehist-cartb %X", i);
        getStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeB[i]);
        sprintf(str, "filehist-diska %X", 0x1882FAF0 + i);
        getStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveA[i]);
        sprintf(str, "filehist-diskb %X", i);
        getStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveB[i]);
    }
}

void propSave(Properties* pProperties) 
{
    int i;
    setIntValue(registryKey, "emulation-family", pProperties->emulation.family);
    setIntValue(registryKey, "emulation-speed", pProperties->emulation.speed);
    setIntValue(registryKey, "emulation-ramsize", pProperties->emulation.RAMsize);
    setIntValue(registryKey, "emulation-vramsize", pProperties->emulation.VRAMsize);
    setIntValue(registryKey, "emulation-syncmethod", pProperties->emulation.syncMethod);
    
    setIntValue(registryKey, "video-montype", pProperties->video.monType);
    setIntValue(registryKey, "video-palemu", pProperties->video.palEmu);
    setIntValue(registryKey, "video-type", pProperties->video.videoType);
    setIntValue(registryKey, "video-size", pProperties->video.size);
    setIntValue(registryKey, "video-driver", pProperties->video.driver);
    setIntValue(registryKey, "video-frameskip", pProperties->video.frameSkip);
    setIntValue(registryKey, "video-fullres", pProperties->video.fullRes);
    
    setIntValue(registryKey, "sound-driver", pProperties->sound.driver);
    setIntValue(registryKey, "sound-frequency", pProperties->sound.frequency);
    setIntValue(registryKey, "sound-bufsize", pProperties->sound.bufSize);
    setIntValue(registryKey, "sound-syncmethod", pProperties->sound.syncMethod);
    setIntValue(registryKey, "sound-logfilecount", pProperties->sound.logFileCount);

	setIntValue(registryKey, "sound-stereo", pProperties->sound.stereo);
    setIntValue(registryKey, "sound-mastervolume", pProperties->sound.masterVolume);
    setIntValue(registryKey, "sound-mcPSGenable", pProperties->sound.mixerChannel[MCT_PSG].enable);
    setIntValue(registryKey, "sound-mcPSGpan", pProperties->sound.mixerChannel[MCT_PSG].pan);
    setIntValue(registryKey, "sound-mcPSGvolume", pProperties->sound.mixerChannel[MCT_PSG].volume);
    setIntValue(registryKey, "sound-mcSCCenable", pProperties->sound.mixerChannel[MCT_SCC].enable);
    setIntValue(registryKey, "sound-mcSCCpan", pProperties->sound.mixerChannel[MCT_SCC].pan);
    setIntValue(registryKey, "sound-mcSCCvolume", pProperties->sound.mixerChannel[MCT_SCC].volume);
    setIntValue(registryKey, "sound-mcAUDIOenable", pProperties->sound.mixerChannel[MCT_MSXAUDIO].enable);
    setIntValue(registryKey, "sound-mcAUDIOpan", pProperties->sound.mixerChannel[MCT_MSXAUDIO].pan);
    setIntValue(registryKey, "sound-mcAUDIOvolume", pProperties->sound.mixerChannel[MCT_MSXAUDIO].volume);
    setIntValue(registryKey, "sound-mcMUSICenable", pProperties->sound.mixerChannel[MCT_MSXMUSIC].enable);
    setIntValue(registryKey, "sound-mcMUSICpan", pProperties->sound.mixerChannel[MCT_MSXMUSIC].pan);
    setIntValue(registryKey, "sound-mcMUSICvolume", pProperties->sound.mixerChannel[MCT_MSXMUSIC].volume);
    setIntValue(registryKey, "sound-mcKEYBOARDenable", pProperties->sound.mixerChannel[MCT_KEYBOARD].enable);
    setIntValue(registryKey, "sound-mcKEYBOARDpan", pProperties->sound.mixerChannel[MCT_KEYBOARD].pan);
    setIntValue(registryKey, "sound-mcKEYBOARDvolume", pProperties->sound.mixerChannel[MCT_KEYBOARD].volume);
    setIntValue(registryKey, "sound-mcCASETTEenable", pProperties->sound.mixerChannel[MCT_CASETTE].enable);
    setIntValue(registryKey, "sound-mcCASETTEpan", pProperties->sound.mixerChannel[MCT_CASETTE].pan);
    setIntValue(registryKey, "sound-mcCASETTEvolume", pProperties->sound.mixerChannel[MCT_CASETTE].volume);

    setIntValue(registryKey, "joy1-type", pProperties->joy1.type);
    setIntValue(registryKey, "joy1-autofire", pProperties->joy1.autofire);
    setIntValue(registryKey, "joy1-keyup", pProperties->joy1.keyUp);
    setIntValue(registryKey, "joy1-keydown", pProperties->joy1.keyDown);
    setIntValue(registryKey, "joy1-keyleft", pProperties->joy1.keyLeft);
    setIntValue(registryKey, "joy1-keyright", pProperties->joy1.keyRight);
    setIntValue(registryKey, "joy1-button1", pProperties->joy1.button1);
    setIntValue(registryKey, "joy1-button2", pProperties->joy1.button2);
    
    setIntValue(registryKey, "joy2-type", pProperties->joy2.type);
    setIntValue(registryKey, "joy2-autofire", pProperties->joy2.autofire);
    setIntValue(registryKey, "joy2-keyup", pProperties->joy2.keyUp);
    setIntValue(registryKey, "joy2-keydown", pProperties->joy2.keyDown);
    setIntValue(registryKey, "joy2-keyleft", pProperties->joy2.keyLeft);
    setIntValue(registryKey, "joy2-keyright", pProperties->joy2.keyRight);
    setIntValue(registryKey, "joy2-button1", pProperties->joy2.button1);
    setIntValue(registryKey, "joy2-button2", pProperties->joy2.button2);
    
    setIntValue(registryKey, "keyboard-keyset", pProperties->keyboard.keySet);
    
    setStrValue(registryKey, "cartridge-slota", (char*)pProperties->cartridge.slotA);
    setStrValue(registryKey, "cartridge-slotb", (char*)pProperties->cartridge.slotB);
    setStrValue(registryKey, "diskdrive-slota", (char*)pProperties->diskdrive.slotA);
    setStrValue(registryKey, "diskdrive-slotb", (char*)pProperties->diskdrive.slotB);
    setIntValue(registryKey, "diskdrive-autostarta", pProperties->diskdrive.autostartA);
    
    for (i = 0; i < MAX_HISTORY; i++) {
        char str[32];
        sprintf(str, "filehist-carta %X", i);
        setStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeA[i]);
        sprintf(str, "filehist-cartb %X", i);
        setStrValue(registryKey, str, (char*)pProperties->filehistory.cartridgeB[i]);
        sprintf(str, "filehist-diska %X", i);
        setStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveA[i]);
        sprintf(str, "filehist-diskb %X", i);
        setStrValue(registryKey, str, (char*)pProperties->filehistory.diskdriveB[i]);
    }
}

Properties* propCreate() 
{
    Properties* pProperties;
    DWORD test = 0;

    setRegIntValue(registryKey, "testval", 42);
    getRegIntValue(registryKey, "testval", &test);

    canUseRegistry = test == 42;

    pProperties = malloc(sizeof(Properties));
    pProperties->joy1.id = 1;
    pProperties->joy2.id = 2;
    propInitDefaults(pProperties);
    propLoad(pProperties);

    return pProperties;
}


void propDestroy(Properties* pProperties) {
    propSave(pProperties);

    free(pProperties);
}

BOOL propSetVideoSize(Properties* pProperties, PropVideoSize size) {
    BOOL changed = pProperties->video.size != size;
    pProperties->video.size = size;
    return changed;
}
