/*****************************************************************************
** File:        videoRender.c
**
** Author:      Daniel Vik
**
** Description: Video render methods. 
**
** License:     Freeware. Anyone may distribute, use and modify the file 
**              without notifying the author. Even though it is not a 
**              requirement, the autor will be happy if you mention his 
**              name when using the file as is or in modified form.
**
** History:     1.0 - 10/17 2003 Initial version
**              1.1 - 11/26 2003 Added delay and glow in pal 32 rendering.
**              1.2 - 12/01 2003 Added double src width support (for MSX
**                               screen 6,7 and text80).
**              1.3 / 12/05 2003 Added support for scale2x (by SLotman)
**
******************************************************************************
*/
#include "videoRender.h"
#include "scalebit.h"
#include <stdlib.h>
#include <stdio.h>
#include <windows.h>

#define YCBCR_MASK 0xFBEF
#define MAX_YCBCR_VALUE (1 << 16)

static UInt32 pRgbTableColor32[MAX_YCBCR_VALUE];
static UInt32 pRgbTableGreen32[MAX_YCBCR_VALUE];
static UInt32 pRgbTableWhite32[MAX_YCBCR_VALUE];
static UInt16 pRgbTableColor16[MAX_YCBCR_VALUE];
static UInt16 pRgbTableGreen16[MAX_YCBCR_VALUE];
static UInt16 pRgbTableWhite16[MAX_YCBCR_VALUE];

UInt32 YJKtoYCbCrTable[32][32][32];

static UInt32 history[240][640];

struct Video {
    UInt16* pRgbTable16;
    UInt32* pRgbTable32;
    VideoPalMode palMode;
    UInt32 decay;
};

#define ABS(a) ((a) < 0 ? -1 * (a) : (a))
#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

static void initYJKtoYCbCrTable() {
    int Y;
    int J;
    int K;

    for (Y = 0; Y < 32; Y++) {
        for (J = -16; J < 16; J++) {
            for (K = -16; K < 16; K++) {
                int Y2, Cb, Cr;
                int R=(Y+2*J);
                int G=(Y+2*K);
                int B=((5*Y-4*J-2*K)/4);
                UInt32 YCbCr;

                Y2 = (int)(0.2989*R + 0.5866*G + 0.1145*B);
                Cb = B - Y2;
                Cr = R - Y2;

                Y2=Y2<0? 0:Y2>31? 31:Y2;
                Cb=Cb<-32? -32:Cb>31? 31:Cb;
                Cr=Cr<-32? -32:Cr>31? 31:Cr;

                YCbCr = (Y2 / 2) | ((16 + (Cb / 2)) << 5) | ((16 + (Cr / 2)) << 11);
                
                YJKtoYCbCrTable[Y][J < 0 ? J + 32 : J][K < 0 ? K + 32 : K] = YCbCr;
            }
        }
    }
}

static void initRGBTable() 
{
    int YCbCr;

    for (YCbCr = 0; YCbCr < MAX_YCBCR_VALUE; YCbCr++) {
        if (pRgbTableColor32[YCbCr] == 0) {
            int Y  = 8 + 16 * (YCbCr & 0x0f);
            int Cb = 8 + 16 * (((YCbCr >> 5) & 0x1f) - 16);
            int Cr = 8 + 16 * (((YCbCr >> 11) & 0x1f) - 16);

            int R = Cr + Y;
            int G = (int)(Y - (0.1145/0.5866)*Cb - (0.2989/0.5866)*Cr);
            int B = Cb + Y;
            int L = Y;
            
            L = MAX(16, MIN(255, (int)(1.12 * L)));
            R = MAX(16, MIN(255, (int)(1.12 * R)));
            G = MAX(16, MIN(255, (int)(1.12 * G)));
            B = MAX(16, MIN(255, (int)(1.12 * B)));

            pRgbTableColor32[YCbCr] = (R << 16) | (G << 8) | (B << 0);
            pRgbTableColor16[YCbCr] = ((R >> 3) << 11) | ((G >> 2) << 5) | (B >> 3);

            pRgbTableGreen32[YCbCr] = 0x100010 | (L << 8);
            pRgbTableGreen16[YCbCr] = 0x0801 | (UInt16)((L >> 2) << 5);

            pRgbTableWhite32[YCbCr] = (L << 16) | (L << 8) | (L << 0);
            pRgbTableWhite16[YCbCr] = (UInt16)(((L >> 3) << 11) | ((L >> 2) << 5) | (L >> 3));
        }
    }
}


/*****************************************************************************
**
** PAL emulation rendering routines
**
******************************************************************************
*/
static void copySharpPAL_2x2_16(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                                int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst1       = (UInt16*)pDestination;
    UInt16* pDst2       = pDst1 + dstPitch / sizeof(UInt16);
    UInt16* pDst3       = pDst2;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev1 = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 colNext1;
                UInt16 colRgb1;
                UInt16 colRgb2;
                UInt32 colTmp;
                UInt16 noise;

                colNext1 = pSrc[w++];
                colTmp   = ((colNext1   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb1  = pRgbTable16[colTmp];

                colPrev1  = colCur;
                colCur    = colNext1;

                colNext1 = pSrc[w++];
                colTmp  = ((colNext1   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb2 = pRgbTable16[colTmp];

                colPrev1 = colCur;
                colCur   = colNext1;

                noise = (UInt16)(rnd >> 31) * 0x0821;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb1 >> 3) & 0x18e3)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb2 >> 3) & 0x18e3)) - noise;
                dstIndex++;

                rnd *= 23;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            int dstIndex = 0;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext;
                UInt16 colRgb1;
                UInt16 colRgb2;
                UInt32 colTmp;
                UInt16 noise;

                colNext = pSrc[w];
                colTmp  = ((colNext + colCur) >> 1) & YCBCR_MASK;
                colRgb1 = pRgbTable16[((colTmp + colCur) >> 1) & YCBCR_MASK];
                colRgb2 = pRgbTable16[((colTmp + colNext) >> 1) & YCBCR_MASK];

                noise = (UInt16)(rnd >> 31) * 0x0821;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb1 >> 3) & 0x18e3)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb2 >> 3) & 0x18e3)) - noise;
                dstIndex++;

                rnd *= 23;
                colCur = colNext;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*) ((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*) ((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void copyPAL_2x2_16(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst1       = (UInt16*)pDestination;
    UInt16* pDst2       = pDst1 + dstPitch / sizeof(UInt16);
    UInt16* pDst3       = pDst2;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev1 = colCur;
            UInt32 colPrev2 = colCur;
            UInt32 colNext1 = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 colNext2;
                UInt16 colRgb1;
                UInt16 colRgb2;
                UInt32 colTmp;
                UInt16 noise;

                colNext2 = pSrc[w++];
                colTmp   = ((colNext2 + colPrev2) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colNext1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb1  = pRgbTable16[colTmp];

                colPrev2  = colPrev1;
                colPrev1  = colCur;
                colCur    = colNext1;
                colNext1  = colNext2;

                colNext2 = pSrc[w++];
                colTmp   = ((colNext2 + colPrev2) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colNext1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb2  = pRgbTable16[colTmp];

                colPrev2  = colPrev1;
                colPrev1 = colCur;
                colCur   = colNext1;
                colNext1  = colNext2;

                noise = (UInt16)(rnd >> 31) * 0x0821;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb1 >> 3) & 0x18e3)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb2 >> 3) & 0x18e3)) - noise;
                dstIndex++;

                rnd *= 23;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext;
                UInt16 colRgb1;
                UInt16 colRgb2;
                UInt32 colTmp;
                UInt16 noise;

                colNext = pSrc[w];

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colPrev) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;
                colRgb1 = pRgbTable16[colTmp];

                colTmp  = ((colNext + colPrev) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;
                colRgb2 = pRgbTable16[colTmp];

                noise = (UInt16)(rnd >> 31) * 0x0821;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb1 >> 3) & 0x18e3)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 3 * (((pDst3[dstIndex] >> 3) & 0x18e3) + ((colRgb2 >> 3) & 0x18e3)) - noise;
                dstIndex++;

                rnd *= 23;

                colCur = colNext;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*) ((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*) ((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void copyPAL_1x1_16(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst        = (UInt16*)pDestination;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcPitch *= 2;

        for (h = 0; h < srcHeight / 2; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = ((pSrc[2 * w] + pSrc[2 * w + 1]) >> 1) & YCBCR_MASK;
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = pRgbTable16[colTmp] - (UInt16)(rnd >> 29)  * 0x0821;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*) ((UInt8*)pDst + dstPitch);

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = ((pSrc[2 * w] + pSrc[2 * w + 1]) >> 1) & YCBCR_MASK;
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = 3 * ((pRgbTable16[colTmp] >> 2) & 0x39e7) - (UInt16)(rnd >> 30) * 0x0821;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*) ((UInt8*)pDst + dstPitch);
        }
    }
    else {       
        for (h = 0; h < srcHeight / 2; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = pSrc[w];
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = pRgbTable16[colTmp] - (UInt16)(rnd >> 29)  * 0x0821;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*) ((UInt8*)pDst + dstPitch);

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = pSrc[w];
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = 3 * ((pRgbTable16[colTmp] >> 2) & 0x39e7) - (UInt16)(rnd >> 30) * 0x0821;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*) ((UInt8*)pDst + dstPitch);
        }
    }
}

static void copySharpPAL_2x2_32(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                                int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst1       = (UInt32*)pDestination;
    UInt32* pDst2       = pDst1 + dstPitch / sizeof(UInt32);
    UInt32* pDst3       = pDst2;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev1 = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 colNext1;
                UInt32 colRgb1;
                UInt32 colRgb2;
                UInt32 colTmp;
                UInt32 noise;

                colNext1 = pSrc[w++];
                colTmp   = ((colNext1   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp   = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb1  = pRgbTable32[colTmp];

                colPrev1  = colCur;
                colCur    = colNext1;

                colNext1 = pSrc[w++];
                colTmp  = ((colNext1   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colCur) >> 1) & YCBCR_MASK;
                colRgb2 = pRgbTable32[colTmp];

                colPrev1 = colCur;
                colCur   = colNext1;

                noise = (rnd >> 30) * 0x10101;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb1 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb2 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;

                rnd *= 23;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            int dstIndex = 0;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext;
                UInt32 colRgb1;
                UInt32 colRgb2;
                UInt32 colTmp;
                UInt32 noise;

                colNext = pSrc[w];
                colTmp  = ((colNext + colCur) >> 1) & YCBCR_MASK;
                colRgb1 = pRgbTable32[((colTmp + colCur) >> 1) & YCBCR_MASK];
                colRgb2 = pRgbTable32[((colTmp + colNext) >> 1) & YCBCR_MASK];

                noise = (rnd >> 30) * 0x10101;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 7 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb1 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 7 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb2 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;

                rnd *= 23;
                colCur = colNext;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void copyPAL_2x2_32(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable, UInt32 decay)
{
    UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst1       = (UInt32*)pDestination;
    UInt32* pDst2       = pDst1 + dstPitch / sizeof(UInt32);
    UInt32* pDst3       = pDst2;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev2 = colCur;
            UInt32 colPrev1 = colCur;
            UInt32 colNext1 = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 colNext2;
                UInt32 colRgb1;
                UInt32 colRgb2;
                UInt32 colTmp;
                UInt32 colLgt;
                UInt32 noise;

                colNext2 = pSrc[w];

                colLgt  = colCur & 0x0f;
                noise = history[h][w] - colLgt;
                colLgt += ((-noise >> 24) & (decay * noise / 5)) - ((noise >> 24) & (decay * -noise / 16));
                history[h][w] = colLgt;
                colLgt += colCur & 0xfffffff0;
                w++;

                colTmp  = ((colPrev2 + colNext2) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colNext1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colLgt) >> 1) & YCBCR_MASK;
                colRgb1 = pRgbTable32[colTmp];

                colPrev2 = colPrev1;
                colPrev1 = colCur;
                colCur   = colNext1;
                colNext1 = colNext2;
                colNext2 = pSrc[w];

                colLgt  = colCur & 0x0f;
                noise = history[h][w] - colLgt;
                colLgt += ((-noise >> 24) & (decay * noise / 5)) - ((noise >> 24) & (decay * -noise / 16));
                history[h][w] = colLgt;
                colLgt += colCur & 0xfffffff0;
                w++;

                colTmp  = ((colPrev2 + colNext2) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colPrev1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colNext1) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp   + colLgt) >> 1) & YCBCR_MASK;
                colRgb2 = pRgbTable32[colTmp];

                colPrev2 = colPrev1;
                colPrev1 = colCur;
                colCur   = colNext1;
                colNext1 = colNext2;

                noise = (rnd >> 30) * 0x10101;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb1 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb2 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;

                rnd *= 23;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;
            int dstIndex = 0;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext;
                UInt32 colRgb1;
                UInt32 colRgb2;
                UInt32 colTmp;
                UInt32 colLgt;
                UInt32 noise;

                colNext = pSrc[w];

                colLgt  = colCur & 0x0f;
                noise = history[h][w] - colLgt;
                colLgt += ((-noise >> 24) & (decay * noise / 5)) - ((noise >> 24) & (decay * -noise / 16));
                history[h][w] = colLgt;
                colLgt += colCur & 0xfffffff0;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colPrev) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colLgt) >> 1) & YCBCR_MASK;
                colRgb1 = pRgbTable32[colTmp];

                colTmp  = ((colNext + colPrev) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colLgt) >> 1) & YCBCR_MASK;
                colRgb2 = pRgbTable32[colTmp];

                noise = (rnd >> 30) * 0x10101;
                pDst2[dstIndex] = colRgb1 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb1 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;
                pDst2[dstIndex] = colRgb2 - noise;
                pDst1[dstIndex] = 6 * (((pDst3[dstIndex] >> 4) & 0x0f0f0f) + ((colRgb2 >> 4) & 0x0f0f0f)) - noise;
                dstIndex++;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pDst3 = pDst2;
            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void copyPAL_1x1_32(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst        = (UInt32*)pDestination;
    int w;
    int h;

    if (srcDoubleWidth) {
        srcPitch *= 2;

        for (h = 0; h < srcHeight / 2; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = ((pSrc[2 * w] + pSrc[2 * w + 1]) >> 1) & YCBCR_MASK;
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = pRgbTable32[colTmp] - (rnd >> 29)  * 0x10101;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = ((pSrc[2 * w] + pSrc[2 * w + 1]) >> 1) & YCBCR_MASK;
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = 7 * ((pRgbTable32[colTmp] >> 3) & 0x1f1f1f) - (rnd >> 30) * 0x10101;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);
        }
    }
    else {
        for (h = 0; h < srcHeight / 2; h++) {
            UInt32 colCur = pSrc[0];
            UInt32 colPrev = colCur;

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = pSrc[w];
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = pRgbTable32[colTmp] - (rnd >> 29)  * 0x10101;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);

            for (w = 0; w < srcWidth; w++) {
                UInt32 colNext = pSrc[w];
                UInt32 colTmp;

                colTmp  = ((colPrev + colNext) >> 1) & YCBCR_MASK;
                colTmp  = ((colTmp + colCur) >> 1) & YCBCR_MASK;

                pDst[w] = 7 * ((pRgbTable32[colTmp] >> 3) & 0x1f1f1f) - (rnd >> 30) * 0x10101;

                rnd *= 23;
                colPrev = colCur;
                colCur = colNext;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);
        }
    }
}


/*****************************************************************************
**
** Fast rendering routines
**
******************************************************************************
*/
static void copy_1x1_16(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                        int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst        = (UInt16*)pDestination;
    int w;
    int h;

    srcPitch -= srcWidth * sizeof(UInt32);
    dstPitch -= srcWidth * sizeof(UInt16);
    srcWidth /= 8;

    if (srcDoubleWidth) {
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            for (w = 0; w < srcWidth; w++) {
                pDst[0] = pRgbTable16[((pSrc[0] + pSrc[1]) >> 1) & YCBCR_MASK];
                pDst[1] = pRgbTable16[((pSrc[2] + pSrc[3]) >> 1) & YCBCR_MASK];
                pDst[2] = pRgbTable16[((pSrc[4] + pSrc[5]) >> 1) & YCBCR_MASK];
                pDst[3] = pRgbTable16[((pSrc[6] + pSrc[7]) >> 1) & YCBCR_MASK];
                pDst[4] = pRgbTable16[((pSrc[8] + pSrc[9]) >> 1) & YCBCR_MASK];
                pDst[5] = pRgbTable16[((pSrc[10] + pSrc[11]) >> 1) & YCBCR_MASK];
                pDst[6] = pRgbTable16[((pSrc[12] + pSrc[13]) >> 1) & YCBCR_MASK];
                pDst[7] = pRgbTable16[((pSrc[14] + pSrc[15]) >> 1) & YCBCR_MASK];
                
                pSrc += 16;
                pDst += 8;
            }
            
            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*)((UInt8*)pDst + dstPitch);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            for (w = 0; w < srcWidth; w ++) {
                pDst[0] = pRgbTable16[pSrc[0]];
                pDst[1] = pRgbTable16[pSrc[1]];
                pDst[2] = pRgbTable16[pSrc[2]];
                pDst[3] = pRgbTable16[pSrc[3]];
                pDst[4] = pRgbTable16[pSrc[4]];
                pDst[5] = pRgbTable16[pSrc[5]];
                pDst[6] = pRgbTable16[pSrc[6]];
                pDst[7] = pRgbTable16[pSrc[7]];
                
                pSrc += 8;
                pDst += 8;
            }

            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt16*) ((UInt8*)pDst + dstPitch);
        }
    }
}

static void copy_2x2_16(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                        int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst1       = (UInt16*)pDestination;
    UInt16* pDst2       = pDst1 + dstPitch / sizeof(UInt16);
    int w;
    int h;    

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;
        for (h = 0; h < srcHeight; h++) {
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt16 col1 = pRgbTable16[pSrc[w++]];
                UInt16 col2 = pRgbTable16[pSrc[w++]];
                UInt16 col3 = pRgbTable16[pSrc[w++]];
                UInt16 col4 = pRgbTable16[pSrc[w++]];

                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
            }

            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            int dstIndex = 0;
            for (w = 0; w < srcWidth;) {
                UInt16 col1 = pRgbTable16[pSrc[w++]];
                UInt16 col2 = pRgbTable16[pSrc[w++]];
                UInt16 col3 = pRgbTable16[pSrc[w++]];
                UInt16 col4 = pRgbTable16[pSrc[w++]];

                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
            }

            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt16*) ((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt16*) ((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void copy_1x1_32(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                        int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst        = (UInt32*)pDestination;
    int w;
    int h;

    srcPitch -= srcWidth * sizeof(UInt32);
    dstPitch -= srcWidth * sizeof(UInt32);
    srcWidth /= 8;

    if (srcDoubleWidth) {
        srcPitch *= 2;

        for (h = 0; h < srcHeight; h++) {
            for (w = 0; w < srcWidth; w++) {
                pDst[0] = pRgbTable32[((pSrc[0] + pSrc[1]) >> 1) & YCBCR_MASK];
                pDst[1] = pRgbTable32[((pSrc[2] + pSrc[3]) >> 1) & YCBCR_MASK];
                pDst[2] = pRgbTable32[((pSrc[4] + pSrc[5]) >> 1) & YCBCR_MASK];
                pDst[3] = pRgbTable32[((pSrc[6] + pSrc[7]) >> 1) & YCBCR_MASK];
                pDst[4] = pRgbTable32[((pSrc[8] + pSrc[9]) >> 1) & YCBCR_MASK];
                pDst[5] = pRgbTable32[((pSrc[10] + pSrc[11]) >> 1) & YCBCR_MASK];
                pDst[6] = pRgbTable32[((pSrc[12] + pSrc[13]) >> 1) & YCBCR_MASK];
                pDst[7] = pRgbTable32[((pSrc[14] + pSrc[15]) >> 1) & YCBCR_MASK];
                
                pSrc += 16;
                pDst += 8;
            }
            
            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            for (w = 0; w < srcWidth; w++) {
                pDst[0] = pRgbTable32[pSrc[0]];
                pDst[1] = pRgbTable32[pSrc[1]];
                pDst[2] = pRgbTable32[pSrc[2]];
                pDst[3] = pRgbTable32[pSrc[3]];
                pDst[4] = pRgbTable32[pSrc[4]];
                pDst[5] = pRgbTable32[pSrc[5]];
                pDst[6] = pRgbTable32[pSrc[6]];
                pDst[7] = pRgbTable32[pSrc[7]];
                
                pSrc += 8;
                pDst += 8;
            }
            
            pSrc = (UInt32*)((UInt8*)pSrc + srcPitch);
            pDst = (UInt32*)((UInt8*)pDst + dstPitch);
        }
    }
}

static void copy_2x2_32(void* pSource, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDestination, 
                        int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
    UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst1       = (UInt32*)pDestination;
    UInt32* pDst2       = pDst1 + dstPitch / sizeof(UInt32);
    int w;
    int h;

    if (srcDoubleWidth) {
        srcWidth *= 2;
        srcPitch *= 2;
        for (h = 0; h < srcHeight; h++) {
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 col1 = pRgbTable32[pSrc[w++]];
                UInt32 col2 = pRgbTable32[pSrc[w++]];
                UInt32 col3 = pRgbTable32[pSrc[w++]];
                UInt32 col4 = pRgbTable32[pSrc[w++]];

                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
            }

            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
    else {
        for (h = 0; h < srcHeight; h++) {
            int dstIndex = 0;

            for (w = 0; w < srcWidth;) {
                UInt32 col1 = pRgbTable32[pSrc[w++]];
                UInt32 col2 = pRgbTable32[pSrc[w++]];
                UInt32 col3 = pRgbTable32[pSrc[w++]];
                UInt32 col4 = pRgbTable32[pSrc[w++]];

                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                pDst1[dstIndex]   = col1;
                pDst2[dstIndex++] = col1;
                
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                pDst1[dstIndex]   = col2;
                pDst2[dstIndex++] = col2;
                
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                pDst1[dstIndex]   = col3;
                pDst2[dstIndex++] = col3;
                
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
                pDst1[dstIndex]   = col4;
                pDst2[dstIndex++] = col4;
            }

            pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
            pDst1 = (UInt32*)((UInt8*)pDst1 + dstPitch * 2);
            pDst2 = (UInt32*)((UInt8*)pDst2 + dstPitch * 2);
        }
    }
}

static void scale2x_2x2_32(void* pSource, int srcWidth, int srcHeight, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{
	UInt32  ImgSrc[80000];
	UInt32* pRgbTable32 = (UInt32*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt32* pDst1       = (UInt32*)pDestination;
	int w, h;    

	srcPitch/=4;

    for (h=0; h<srcHeight; h++) 
	{
        for (w=0; w<srcWidth; w++) 
		{
			ImgSrc[w+h*srcWidth]=pRgbTable32[pSrc[w]];
        }
		pSrc+=srcPitch;
    }
    scale(2, &pDst1[0], srcWidth*8, &ImgSrc[0], srcWidth*4, 4, srcWidth, srcHeight);
}

static void scale2x_2x2_16(void* pSource, int srcWidth, int srcHeight, void* pDestination, 
                           int srcPitch, int dstPitch, UInt32 rnd, void* pRgbTable)
{

    UInt16* pRgbTable16 = (UInt16*)pRgbTable;
    UInt32* pSrc        = (UInt32*)pSource;
    UInt16* pDst1       = (UInt16*)pDestination;
	UInt16  ImgSrc[320*250];
	UInt16* ImgDst;
	int w, h,index;    

    index=0;
    for (h=0; h<srcHeight; h++) 
	{
        for (w=0; w<srcWidth; w++) 
		{
			ImgSrc[index]=pRgbTable16[pSrc[w]];
			index++;
        }
        pSrc  = (UInt32*)((UInt8*)pSrc  + srcPitch);
    }
    scale(2, &pDst1[0], srcWidth*4, &ImgSrc[0], srcWidth*2, 2, srcWidth, srcHeight);
}

/*****************************************************************************
**
** Public interface methods
**
******************************************************************************
*/
Video* videoCreate() 
{
    Video* pVideo = (Video*)calloc(sizeof(Video), 1);
    initYJKtoYCbCrTable();
    initRGBTable();

    pVideo->palMode = VIDEO_PAL_FAST;
    pVideo->pRgbTable16 = pRgbTableColor16;
    pVideo->pRgbTable32 = pRgbTableColor32;

    return pVideo;
}

void videoDestroy(Video* pVideo) 
{
    free(pVideo);
}

void videoSetFrameSkip(Video* pVideo, UInt32 skipCount)
{
    if (skipCount > 3) skipCount = 3;
    pVideo->decay = 3 - skipCount;
}

UInt32 videoGetColor(Video* pVideo, int R, int G, int B)
{
    /* Get YCbCr color from an RGB color. The color tables are updated with
     * the exact color.
     */

    int Y  = (int)(0.2989*R + 0.5866*G + 0.1145*B);
    int Cb = B - Y;
    int Cr = R - Y;
    int L  = Y;
    UInt32 YCbCr = (Y / 16) | ((16 + (Cb / 16)) << 5) | ((16 + (Cr / 16)) << 11);

    L = MAX(16, MIN(255, L));
    R = MAX(16, MIN(255, R));
    G = MAX(16, MIN(255, G));
    B = MAX(16, MIN(255, B));

    pRgbTableColor32[YCbCr] = ((R << 16) | (G << 8) | B);
    pRgbTableGreen32[YCbCr] = 0x100010 | (L << 8);
    pRgbTableWhite32[YCbCr] = (L << 16) | (L << 8) | (L << 0);

    pRgbTableColor16[YCbCr] = ((R >> 3) << 11) | ((G >> 2) << 5) | (B >> 3);
    pRgbTableGreen16[YCbCr] = 0x0801 | (UInt16)((L >> 2) << 5);
    pRgbTableWhite16[YCbCr] = (UInt16)(((L >> 3) << 11) | ((L >> 2) << 5) | (L >> 3));

    return YCbCr;
}

void videoSetColorMode(Video* pVideo, VideoColorMode colorMode) 
{
    static int initialized = 0;

    if (!initialized) {
        initYJKtoYCbCrTable();
        initRGBTable();
        initialized = 1;
    }

    switch (colorMode) {
    case VIDEO_GREEN:
        pVideo->pRgbTable16 = pRgbTableGreen16;
        pVideo->pRgbTable32 = pRgbTableGreen32;
        break;
    case VIDEO_BLACKWHITE:
        pVideo->pRgbTable16 = pRgbTableWhite16;
        pVideo->pRgbTable32 = pRgbTableWhite32;
        break;
    case VIDEO_COLOR:
    default:
        pVideo->pRgbTable16 = pRgbTableColor16;
        pVideo->pRgbTable32 = pRgbTableColor32;
        break;
    }
}

void videoSetPalMode(Video* pVideo, VideoPalMode palMode)
{
    pVideo->palMode = palMode;
}


void videoRender(Video* pVideo, int bitDepth, int zoom, 
                 void* pSrc, int srcWidth, int srcHeight, int srcDoubleWidth, void* pDst, int srcPitch, int dstPitch)
{
    static UInt32 rnd = 51;

    // Update simple rand generator
    rnd *= 13;

    switch (bitDepth) {
    case 16:
        switch (pVideo->palMode) {
        case VIDEO_PAL_FAST:
            if (zoom == 2) copy_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            else           copy_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            break;
        case VIDEO_PAL_SHARP:
            if (zoom == 2) copySharpPAL_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            else           copyPAL_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            break;
        case VIDEO_PAL_SHARP_NOISE:
            if (zoom == 2) copySharpPAL_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable16);
            else           copyPAL_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable16);
            break;
        case VIDEO_PAL_BLUR:
            if (zoom == 2) copyPAL_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            else           copyPAL_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            break;
        case VIDEO_PAL_BLUR_NOISE:
            if (zoom == 2) copyPAL_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable16);
            else           copyPAL_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable16);
            break;
		case VIDEO_PAL_SCALE2X:
            if (zoom==2) {
                if (!srcDoubleWidth) {
                    scale2x_2x2_16(pSrc, srcWidth, srcHeight, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
                }
                else {
                    copy_2x2_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
                }
            }
            else {
                copy_1x1_16(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable16);
            }
            break;
        }
        break;
    case 32:
        switch (pVideo->palMode) {
        case VIDEO_PAL_FAST:
            if (zoom == 2) copy_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            else           copy_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            break;
        case VIDEO_PAL_SHARP:
            if (zoom == 2) copySharpPAL_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            else           copyPAL_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            break;
        case VIDEO_PAL_SHARP_NOISE:
            if (zoom == 2) copySharpPAL_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable32);
            else           copyPAL_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable32);
            break;
        case VIDEO_PAL_BLUR:
            if (zoom == 2) copyPAL_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32, pVideo->decay);
            else           copyPAL_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            break;
        case VIDEO_PAL_BLUR_NOISE:
            if (zoom == 2) copyPAL_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable32, pVideo->decay);
            else           copyPAL_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, rnd, pVideo->pRgbTable32);
            break;
		case VIDEO_PAL_SCALE2X:
            if (zoom==2) {
                if (!srcDoubleWidth) {
                    scale2x_2x2_32(pSrc, srcWidth, srcHeight, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
                }
                else {
                    copy_2x2_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
                }
            }
            else {
                copy_1x1_32(pSrc, srcWidth, srcHeight, srcDoubleWidth, pDst, srcPitch, dstPitch, 0, pVideo->pRgbTable32);
            }
            break;
        }
        break;
    }
}

